#include <iostream>
#include <cstdio>
#include <distribution/mpi/mpi_control.h>
#include <distribution/mpi/mpi_platypus_types.h>

namespace Platypus
{

  MPIControl::~MPIControl()
  {
    delete pa_;
    delete modelHandler_;
    delete exitHandler_;
    delete stateHandler_;
    delete serializer_;
  }
  
  MPIControl::MPIControl(const ProgramInterface& program,
			 MPIStatistics& statistics,
			 AnswerSetPrinterBase& printer, 
			 size_t requested_answersets,
			 bool suppressed)
    
    :   program_(&program)
    ,   serializer_(new MPISerializer(program))
    ,   statistics_(&statistics)
    ,	printer_(&printer)
    ,   stateHandler_(new MPIControlStateHandler())
    ,   exitHandler_(new MPIControlExitHandler())
    ,   modelHandler_(new MPIControlModelHandler())
    ,   pa_(new PartialAssignment(program))
    ,	requestedAnswerSets_(requested_answersets)
    ,	receivedAnswerSets_(0)
    ,   globalAnswersFound_(0)
    ,   suppressed_(suppressed)
    ,   requests_(0)
    ,	shutdown_(false)
    ,   workers_(0)
    ,   answersFoundByWorker_(0)
    ,   start_time_(0)
  {}
  
  void MPIControl::setup()
  {
    start_time_ = MPI::Wtime();
    workers_ = MPI::COMM_WORLD.Get_size() - 1;
    stateHandler_->setup(workers_);
    exitHandler_->setup(workers_);
    modelHandler_->setup(workers_, requestedAnswerSets_);

    //set the buffer size for the two message buffers
    //size_t bufferSize = serializer_->bytesRequired();
    //paBuffer_.resize(bufferSize);
    //dcBuffer_.resize(bufferSize);
    
    //bootstrap by putting a choice into the queue
    DelegatableChoice* dc = new DelegatableChoice();
    choices_.push(dc);
    
  }

  void MPIControl::initiateTermination(){
    
    char dummy = 0;

    //send terminate message to all workers
    for(unsigned i = 0; i < workers_; i++)
      {
	int messageDest = i+1;
	MPI::COMM_WORLD.Send(&dummy, 
			     1, 
			     MPI::CHAR, 
			     messageDest, 
			     TERMINATE_TAG);

	statistics_->incMessagesSent(MASTER);
      }
  }
  
  void MPIControl::completeTermination(){
    
    unsigned terminationsReceived = 0;
    std::vector<unsigned long> buffer(NUM_STATS,0);

    while(terminationsReceived < workers_)
      {

	//non-blocking probe for the first available message
	if(MPI::COMM_WORLD.Iprobe(MPI::ANY_SOURCE, TERMINATE_CONFIRMATION_TAG, localControllerStatus_))
	  {
	    int source = localControllerStatus_.Get_source();
	    
	    MPI::COMM_WORLD.Recv(&buffer[0], 
				 NUM_STATS, 
				 MPI::UNSIGNED_LONG, 
				 source,
				 TERMINATE_CONFIRMATION_TAG);

	    statistics_->incMessagesReceived(MASTER);
	    
	    //gather stats from individual workers
	    statistics_->incExpanderInits(source, buffer[EXPANDER_INITS]);
	    statistics_->incConflicts(source, buffer[CONFLICTS]);
	    statistics_->incModels(source, buffer[MODELS]);
	    statistics_->incBacktracks(source, buffer[BACKTRACKS]);
	    statistics_->incThreadDelegations(source, buffer[DELEGATIONS]);
	    statistics_->incMessagesSent(source, buffer[MESSAGES_SENT]);
	    statistics_->incMessagesReceived(source, buffer[MESSAGES_RECEIVED]);
	    statistics_->incDroppedRequests(source, buffer[DROPPED_REQUESTS]);

	    terminationsReceived++;
	  }
      }
  }

  void MPIControl::requestDCFromDelegatedWorkers(unsigned source)
  {

    //cheap way to get the internal information in the worker state array
    vector<unsigned>* states = stateHandler_->getStates();
    //find the workers with DELEGATED written into their index in the worker
    //state array and send thema a meesage requesting a partial assignment
    for(unsigned i = 0; i < (*states).size(); i++)
      {
	if((*states)[i] == DELEGATED)
	  {
	    //the mpi id is one more than the index into the vector
	    unsigned worker = i+1;
	    char dummy = 0;
	    localControllerRequest_ = MPI::COMM_WORLD.Isend(&dummy,
							    1,
							    MPI::CHAR,
							    worker,
							    DC_NEEDED_TAG);

	    
	    //to avoid deadlock you need to make sure the request message was sent not at the
	    //same time the worker sent a message requesting work or an answer message
	    while(!localControllerRequest_.Test())
	      {
		if(MPI::COMM_WORLD.Iprobe(worker, DC_REQUEST_TAG))
		  {
		    //index into data structure is one less than mpi id
		    stateHandler_->setState(worker-1, FILED);
		    localControllerRequest_.Cancel();
		  }
		if(!suppressed_)
		  {
		    MPI::Status localStatus;
		    if(MPI::COMM_WORLD.Iprobe(worker, ANSWER_MESSAGE_TAG, localStatus))
		      {
			unsigned count = localStatus.Get_count(MPI::UNSIGNED_LONG);
			handleAnswerSet(worker, count);
		      }
		  }
	      }
	    statistics_->incWorkRequestsFromMaster(worker);
	    statistics_->incMessagesSent(MASTER);
	    
	  }
      }
  }

  void MPIControl::handleAnswerSet(unsigned source, unsigned count)
  {

    if(count != paBuffer_.size())
      paBuffer_.resize(count);

    MPI::COMM_WORLD.Recv(&paBuffer_[0], 
			 paBuffer_.size(), 
			 MPI::UNSIGNED_LONG, 
			 source, 
			 ANSWER_MESSAGE_TAG);
    
    
    statistics_->incMessagesReceived(MASTER);
    statistics_->incAnswers();
    modelHandler_->incLocalModels();
    
    //print the answer set ony if answer set printing is not suppresses
    if(!suppressed_)
      {
	serializer_->deserialize(*pa_, paBuffer_);
	print(*pa_);
      }
  }

  void MPIControl::handleDelegatableChoice(unsigned source, unsigned count)
  {

    //this may be hooping things
    if(count != dcBuffer_.size())
      dcBuffer_.resize(count);

    MPI::COMM_WORLD.Recv(&dcBuffer_[0],                                                                                   
			 count,
			 MPI::UNSIGNED_LONG,                                                                              
			 source,                                                                                          
			 DC_TAG_FROM_SLAVE);                                                                              

    statistics_->incMessagesReceived(MASTER);                                                                             
    statistics_->incWorkDelegationsToMaster(source);                                                                      
    statistics_->setLargestDCMessage(count);
    statistics_->incTotalDCMessages();
    statistics_->incTotalDCMessageCount(count);
    

    //if there is a worker request previously queued then send the 
    //received partial assignment to the requesting worker        
    if(filedWorkers_.size())                                                                                              
      {                                                                                                                   
	unsigned workerCandidate = filedWorkers_.front();                                                                 
	MPI::COMM_WORLD.Send(&dcBuffer_[0],                                                                               
			     dcBuffer_.size(),                                                                            
			     MPI::UNSIGNED_LONG,                                                                          
			     workerCandidate,                                                                             
			     DC_TAG_FROM_CONTROLLER);                                                                     
	statistics_->incMessagesSent(MASTER);                                                                             
	filedWorkers_.pop();    

	statistics_->incTotalDCMessages();
	statistics_->incTotalDCMessageCount(count);                                                                                          
	
	//index into data structure is one less than mpi id                                                               
	stateHandler_->setState(workerCandidate-1, DELEGATED);                                                            
	exitHandler_->decRequests();                                                                                      
      }
    //if there is no queued request place it on the local queue of choices
    else                                                                                                                  
      {                                                                                                                   
	DelegatableChoice* newDc = new DelegatableChoice();                                                               
	serializer_->deserialize(*newDc, dcBuffer_);                                                                      
	choices_.push(newDc);                                                                                             
	statistics_->checkMaxQueueSize(choices_.size());                                                                  
      }                                              
                                                                     
  }
  
  void MPIControl::handleDelegatableChoiceRequest(unsigned source)
  {

    unsigned index = source - 1;//index into state manager is one less than mpi id
    answersFoundByWorker_ = 0;                                                                                                           

    MPI::COMM_WORLD.Recv(&answersFoundByWorker_,                                                                                         
			 1 ,                                                                                                             
			 MPI::UNSIGNED_LONG,                                                                                             
			 source,                                                                                                         
			 DC_REQUEST_TAG);                                                                                                

    statistics_->incMessagesReceived(MASTER);                                                                                            
    statistics_->incWorkRequestsToMaster(source);                                                                                        

    //update the global number of answer sets found                                                                                      
    modelHandler_->setGlobalModels(index, answersFoundByWorker_);                                                                        

    //if there is a choice on the local work queue send it to the
    //requesting worker
    if(choices_.size())                                                                                                                  
      {                                                                                                                                  
	//handle the bootstrap case of an empty partial assignment
	//every other case will laready have DELEGATED in the index
	stateHandler_->setState(index, DELEGATED);                                                                                       

	DelegatableChoice * sendDc = choices_.front();                                                                                   
	serializer_->serialize(dcBuffer_, *sendDc);                                                                                      
	MPI::COMM_WORLD.Send(&dcBuffer_[0],                                                                                              
			     dcBuffer_.size(),                                                                                           
			     MPI::UNSIGNED_LONG,                                                                                         
			     source,                                                                                                     
			     DC_TAG_FROM_CONTROLLER);                                                                                    

	statistics_->incMessagesSent(MASTER);                                                                                            
	statistics_->incWorkDelegationsFromMaster(source);                                                                               

	statistics_->incTotalDCMessages();
	statistics_->incTotalDCMessageCount(dcBuffer_.size());  

	delete sendDc;                                                                                                                   
	choices_.pop();                                                                                                                  
      }
    //if there is no local work then the master must initiate the exteral
    //work request procedure
    else                                                                                                                                 
      {                                                                                                                                  
	stateHandler_->setState(index, FILED);                                                                                       
	exitHandler_->incRequests();      
	statistics_->incWorkDenials(source);                                                                                             
	filedWorkers_.push(source);                                                                                                      
	statistics_->checkMaxFiledWorkers(filedWorkers_.size());                                                                         
       
	requestDCFromDelegatedWorkers(source);                                                                                           
      }
  }

  //the loop where worker messages are handled
  void MPIControl::start()
  {
    while(!stop())
      {

	assert(requests_ >= 0);
	
	//blocking probe, no messages than no reason for master to be in a busy wait
	MPI::COMM_WORLD.Probe(MPI::ANY_SOURCE, MPI::ANY_TAG, localControllerStatus_);  
	
	size_t source = localControllerStatus_.Get_source();
	size_t tag = localControllerStatus_.Get_tag();
	size_t count = localControllerStatus_.Get_count(MPI::UNSIGNED_LONG);

	//partial assignment received from worker
	if(tag == DC_TAG_FROM_SLAVE)
	  {
	    handleDelegatableChoice(source, count);
	  }	    
	//partial assignment requested from worker
	else if(tag == DC_REQUEST_TAG)
	  {
	    handleDelegatableChoiceRequest(source);
	  }
	//answer received from worker
	else if(tag == ANSWER_MESSAGE_TAG)
	  {
	    handleAnswerSet(source, count);
	  }
      }

    finish();
  }

  void MPIControl::finish()
  {
    //print all answer sets if applicable
    cleanupAnswerSets(); 
    //send terminate messages to workers
    initiateTermination();
    //wait for worker termination confirmations 
    completeTermination();
    //print all answer sets if applicable
    //cleanupAnswerSets();
    //clean up stray messages 
    cleanup();    
  }

  /* Test for the exit condition. Either the master has received the
   * the number of answers requested or the natural exit condition has
   * been reached.
   */
  bool MPIControl::stop()
  {
    //if(requestedAnswerSets_ == 0)
    //return(exitHandler_->exit() && (modelHandler_->localModels() == modelHandler_->globalModels()));
    //else
    return(exitHandler_->exit() || enoughReceived());
    //return(exitHandler_->exit());
  }

  bool MPIControl::enoughReceived() const
  {
    return modelHandler_->enough();
  }

  void MPIControl::print(PartialAssignment& pa)
  {
    printer_->print(modelHandler_->localModels(), pa);
  }

  void MPIControl::cleanupAnswerSets()
  {
    if(suppressed_)
      return;

    if(requestedAnswerSets_ == 0)
      {
	while(modelHandler_->localModels() != modelHandler_->globalModels())
	  {
	    std::cout << "modelHandler_->localModels(): " << modelHandler_->localModels() << " modelHandler_->globalModels(): " << modelHandler_->globalModels() << std::endl;
	    if(MPI::COMM_WORLD.Iprobe(MPI::ANY_SOURCE, ANSWER_MESSAGE_TAG, localControllerStatus_))
	      {
		std::cout << "modelHandler_->localModels(): " << modelHandler_->localModels() << " modelHandler_->globalModels(): " << modelHandler_->globalModels() << std::endl; 
		unsigned source = localControllerStatus_.Get_source();
		unsigned count = localControllerStatus_.Get_count(MPI::UNSIGNED_LONG);
		handleAnswerSet(source, count);
	      }
	  }
      }
    else
      {
	while(!enoughReceived())
	  {
	    if(MPI::COMM_WORLD.Iprobe(MPI::ANY_SOURCE, ANSWER_MESSAGE_TAG, localControllerStatus_))
	      {
		unsigned source = localControllerStatus_.Get_source();
		unsigned count = localControllerStatus_.Get_count(MPI::UNSIGNED_LONG);
		handleAnswerSet(source, count);
	      }
	  }
      }
  }

  //clean up any stray messages on the MPI mesage queue
  void MPIControl::cleanup() 
  {

    //if(!enoughReceived())
    //{
    //while(receivedAnswerSets_ != globalAnswersFound_)
    //{
    while(MPI::COMM_WORLD.Iprobe(MPI::ANY_SOURCE, MPI::ANY_TAG, localControllerStatus_))
      {
	unsigned source = localControllerStatus_.Get_source();
	unsigned tag = localControllerStatus_.Get_tag();
	//unsigned count = localControllerStatus_.Get_count();
	  
	  if(tag == ANSWER_MESSAGE_TAG)
	    {
	      
	      unsigned count = localControllerStatus_.Get_count(MPI::UNSIGNED_LONG);
	      paBuffer_.resize(count);
	      MPI::COMM_WORLD.Recv(&paBuffer_[0],
				   paBuffer_.size(),
				   MPI::UNSIGNED_LONG,
				   source,
				   ANSWER_MESSAGE_TAG);
	      statistics_->incMessagesReceived(MASTER);
	      
	      /*
		if(!suppressed_)
		{
		serializer_->deserialize(*pa_, paBuffer_);
		print(*pa_);
		}
	      */
	      //unsigned count = localStatus.Get_count(MPI::UNSIGNED_LONG);
	      //handleAnswerSet(worker, count);  
	    }
	  else if(tag == DC_TAG_FROM_SLAVE)
	    {
	      MPI::COMM_WORLD.Recv(&dcBuffer_[0],
				   dcBuffer_.size(),
				   MPI::UNSIGNED_LONG,
				   source,
				   DC_TAG_FROM_SLAVE);
	      statistics_->incMessagesReceived(MASTER);
	    }
	  else if(tag == DC_REQUEST_TAG)
	    {
	      char dummy = 0;
	      MPI::COMM_WORLD.Recv(&dummy,
				   1,
				   MPI::CHAR,
				   source,
				   DC_REQUEST_TAG);
	      statistics_->incMessagesReceived(MASTER);
	    }
      }
    //}
    //}
  }    

}
