/*	$Id: fork.h 1778 2005-05-17 14:37:06Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef PLATYPUS_MPI_H
#define PLATYPUS_MPI_H

#include <iosfwd>
#include <unistd.h>
#include <cassert>
#include <cstdio>
#include <cerrno>
#include <vector>
#include <iostream>
#include <sstream>
#include <nop_stream.h>
#include <platypus/types.h>
#include <platypus/types_fwd.h>
#include <platypus/types/choice.h>
#include <platypus/types/partial_assignment.h>
#include <platypus/types/atom.h>
#include <platypus/options.h>
#include <platypus/exceptions.h>
#include <platypus/answer_set_printer_base.h>
#include <platypus/distribution_base.h>
#include <platypus/factories/distribution_factory.h>
#include <platypus/factories/registration.h>
#include <program_options/program_options.h>
#include <program_options/value.h>
#include <distribution/mpi/platypus_time.h>
#include <distribution/mpi/mpi_control.h>
#include <distribution/mpi/mpi_serializer.h>
#include <distribution/mpi/mpi_statistics.h>
#include <distribution/mpi/mpi_control_model_handler.h>
#include <distribution/mpi/mpi_platypus_types.h>

class NopStream;
namespace Platypus
{
	const char* const MPI_DISTRIBUTION = "mpi";


	/**
	 * Controls the behaviour of workers as well as the general start up and
	 * shut down behaviour of the mpi specific module of Platypus.
	 *
	 */
	class PlatypusMPI  :  public DistributionBase

	{
	public:
		~PlatypusMPI();
		PlatypusMPI();
		static DistributionBase* create();

	private://in all interfaces extended by DistributionBase
		void setup();

	private: // BuilderDistributionCallback

		/**
		 * Process command line arguments.
		 */
		void processCommandLine(int& argc, char** argv);

		/**
		 * Set the globally accessible program representation.
		 */
		void program(const ProgramInterface& prog);

		/**
		 * Set the output stream.
		 */
		void output(NopStream& str);

		/**
		 * Set up any options that may be needed for the run.
		 */
		void options(const PlatypusOptions& values);

	private: // CoreDistributionCallback

		/**
		 * Tests to see if the current run should shut down.
		 * Reasons for shutdown could include a termination message 
		 * from the master or some exception was caught.
		 */ 
		bool shutdown();

		/**
		 * Tests to see if some remote platypus instance needs
		 * a choice from the local Platypus instance.
		 */
		bool needDelegatableChoice();

		/**
		 * Responsible for sending a choice to the master.
		 */
		bool delegate(const DelegatableChoice& dc);

		/**
		 * File a choice request to the master process.
		 */
		void fileDelegatableChoiceRequest();

		/**
		 * Cancels a previously filed choice request.
		 */
		void cancelDelegatableChoiceRequest();

		/**
		 * Returns true if the calling worker has a choice to share.
		 */
		bool hasDelegatableChoice();

		/**
		 * Returns a choice from the local (calling) worker.
		 */
		DelegatableChoice delegatableChoice();

		/**
		 * Handles answer sets appropriately when found. Simply
		 * sends the answer to the master for processing.
		 */
		void answerSet(const PartialAssignment& pa);

		//following methods increment worker statistic counters. 
		void incExpanderInitializations(size_t inc = 1);
		void incConflicts(size_t inc = 1);
		void incBacktracks(size_t inc = 1);
		void incAnswerSets(size_t inc = 1);
		void incThreadDelegations(size_t inc = 1);
		void incMessagesSent(size_t inc = 1);
		void incMessagesReceived(size_t inc = 1);
		void incDroppedRequests(size_t inc = 1);

		/**
		 * Return the id of the worker process.
		 */
		int id() const;
		
	private: // PlatypusAlgorithmDistributionCallback

		/**
		 * Performs the operations necessary for a clean exit from 
		 * the mpi "world". This includes calling MPI::Intracomm::Finalize().
		 */ 
		void teardown();

		/**
		 * Tests to see if this remote instance is responsible for printing
		 * statistics and answer sets.
		 */
		bool print() const;

		/**
		 * Sets the local print stream.
		 */
		std::ostream& print(std::ostream& os) const;

		//Following methods return values for worker statistics.
		size_t expanderInitializations() const;
		size_t conflicts() const;
		size_t answerSetsFound() const;
		size_t backtracks() const;
		size_t threadDelegations() const;
		size_t threads() const;
		size_t messagesSent() const;
		size_t messagesReceived() const;
		size_t droppedRequests() const;

		//Returns info on the current Platypus type.
		const std::string& type() const { return type_; }

		void terminate();

		/**
		 * Set the printer object.
		 */
		void printer(AnswerSetPrinterBase& pr);	

	private://PlatypusMPI methods
		size_t processes() const;

	private://PlatypusMPI specific methods
		void disableAnswerSetPrinting();
		void enableAnswerSetPrinting();
		bool enoughAnswerSets() const;
		bool enoughPrinted() const;
		size_t bytesRequiredPA() const;
		size_t bytesRequiredDC() const;

		//PlatypusMPI specific methods
		/**
		 * Receive a choice from the master.
		 */
		bool receiveDC();

		/**
		 * Send a choice to the master.
		 */
		bool sendDC(const DelegatableChoice& dc);

		/**
		 * Perform mpi specific initializations.
		 */
		void initialize();

		/**
		 * Clean up any stray messages.
		 */
		void cleanup();

	private:
		typedef std::vector<DelegatableChoice> BranchesToDo;

		NopStream* os_;
		AnswerSetPrinterBase* printer_;
		//DelegatableChoice dc_;
		const ProgramInterface* program_;
		MPISerializer* serializer_;
		std::vector<unsigned long>* stats_;
		MPIStatistics* statistics_;
		MPIControl* controller_;

		size_t requestedAnswerSets_;
		size_t threads_;
		size_t messages_;
		bool suppressAnswerSets_;
		bool hasDC_;
		bool shutdown_;
		bool verbose_;

		size_t id_;
		size_t processes_;
		static const std::string type_;

		DelegatableChoice dc_;
		MessageBuffer PABuffer_;
		size_t bufferSizePA_;
		MessageBuffer DCBuffer_;
		size_t bufferSizeDC_;

		MPI::Status localSlaveStatus_;
		MPI::Request localSlaveRequest_;

		double starttime_;
		double endtime_;

	};
}

#endif
