/*	$Id: fork_serializer.cpp 1782 2005-05-17 14:52:01Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <iostream>
#include <mpi.h>
#include <cstdio>
#include <platypus/types/atom.h>
#include <platypus/types/partial_assignment.h>
#include <distribution/mpi/mpi_serializer.h>

const int SIZE = 0;
const int START = 1;

namespace Platypus
{
  MPISerializer::MPISerializer(const ProgramInterface& program)
    :	program_(&program)
  {}
  
  size_t MPISerializer::bytesRequired() const
  {
    return (1 + (program_->numberOfAtoms() * 2));
  }

  bool MPISerializer::deserialize(DelegatableChoice& dc, const MessageBuffer& buffer) const
  {
    const unsigned size = buffer.size();
    dc.clear();

    for(size_t i = 0; i < size; i += 2)
      {
	dc.push_back(Choice(buffer[i], buffer[i+1]));
      }

    return true;
  }

  void MPISerializer::serialize(MessageBuffer& buffer, const DelegatableChoice& dc) const
  {
    //find the size of the dc... it is simply a typedefed std container (vector of choices)
    //this gives the number of choices, each which contains an id and a boolean value. To send 
    //this via a message you need to add both values to the message.
    const size_t size = dc.size();
    buffer.resize(size * 2);

    for(size_t i = 0; i < size; i++)
      {
	unsigned index = i * 2;
	buffer[index] = dc[i].id();	
	buffer[index+1] = dc[i].isPositive();
      }
  }
  
  void MPISerializer::deserialize(PartialAssignment& pa, const MessageBuffer& buffer) const
  {
    const unsigned size = buffer.size();
    pa.clear();
    for(size_t i = 0; i < size; ++i)
      {
	pa.setTrue(Atom(buffer[i], *program_));
      }
  }

  void MPISerializer::serialize(MessageBuffer& buffer, const PartialAssignment& pa) const
  {
    const size_t size = pa.positiveAtoms().size();
    size_t index = 0;

    if(buffer.size() != size)
      buffer.resize(size);
    
    const PartialAssignment::CollectionType::const_iterator END = pa.positiveAtoms().end();
    for(PartialAssignment::CollectionType::const_iterator it = pa.positiveAtoms().begin();
	it != END; ++it, ++index)
      {
	buffer[index] = it->id();
      }
  }
}
