/*	$Id: delegatable_choice_policy.cpp 2709 2005-10-30 09:25:51Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */


#include <cassert>
#include <platypus/factories/registration.h>
#include <platypus/factories/delegatable_choice_policy_factory.h>
#include <platypus/types/search_space.h>
#include <platypus/types/delegatable_choice_policy.h>

using namespace std;

namespace Platypus
{
	template<>
	void factoryRegistration<DelegatableChoicePolicy::Random>()
	{
		DCPolicyFactory::instance().add("random", &DelegatableChoicePolicy::Random::create, "Choose random part of the search space");
	}

	template<>
	void factoryRegistration<DelegatableChoicePolicy::Largest>()
	{
		DCPolicyFactory::instance().add("largest", &DelegatableChoicePolicy::Largest::create, "Choose largest part of the search space");
	}

	template<>
	void factoryRegistration<DelegatableChoicePolicy::Smallest>()
	{
		DCPolicyFactory::instance().add("smallest", &DelegatableChoicePolicy::Smallest::create, "Choose smallest part of the search space");
	}

	namespace DelegatableChoicePolicy
	{
		PolicyBase::PolicyBase()
			:	space_(0)
		{}

		void PolicyBase::setup(SearchSpace& space)
		{
			space_ = &space;
		}
		
		SearchSpace& PolicyBase::searchSpace()
		{
			assert(space_);
			return *space_;
		}
		const SearchSpace& PolicyBase::searchSpace() const
		{
			assert(space_);
			return *space_;
		}

		Random::Random()
			:	rng_(static_cast<unsigned long>(PortableThreads::pt_seed()))
		{}
		DelegatableChoice Random::choose()
		{
			candidates_.clear();
			
			for(iterator beg = searchSpace().leafBegin(), 
				END = searchSpace().leafEnd();
				beg != END; ++beg)
			{
				if(!beg->active())
					candidates_.push_back(beg);
			}

			if(candidates_.empty())
				return DelegatableChoice();
			

			const unsigned long r = rng_.urand() % candidates_.size();
			return searchSpace().extract(candidates_[r]);
		}
		PolicyBase* Random::create()
		{
			return new Random;
		}
		
		DelegatableChoice Largest::choose()
		{
			const SearchSpace::cursor ROOT = searchSpace().root();
			toLookAt_.clear();
			toLookAt_.push_back(ROOT);
			while(!toLookAt_.empty())
			{
				SearchSpace::cursor current = toLookAt_.front();
				toLookAt_.pop_front();
				if(current.leaf() && 
					!current->active())
				{
					assert(current != ROOT);
					return searchSpace().extract(current);
				}
				else
				{
					if(current.has_left_child())
						toLookAt_.push_back(current.left());
					if(current.has_right_child())
						toLookAt_.push_back(current.right());
				}
			}
			return DelegatableChoice();
		}
		PolicyBase* Largest::create()
		{
			return new Largest;
		}

		DelegatableChoice Smallest::choose()
		{
			iterator pick = searchSpace().leafEnd(), END = searchSpace().leafEnd();
			unsigned d = 0;
			for(iterator beg = searchSpace().leafBegin(); beg != END; ++beg)
			{
				if(!beg->active())
				{
					if(pick == END)
					{
						pick = beg;
						d = distance(pick);
					}
					else for(const unsigned other = distance(beg); other > d;)
					{
						pick = beg;
						d = other;
					}
				}
			}

			return pick == END ? DelegatableChoice() : searchSpace().extract(pick);
		}
		unsigned Smallest::distance(iterator it) const
		{
			unsigned d = 0;
			for(iterator root = searchSpace().root(); it != root; it.ascend())
				++d;
			return d;
		}

		PolicyBase* Smallest::create()
		{
			return new Smallest;
		}
		
	}
}
