#ifndef MPI_PLATYPUS_CONTROL_H
#define MPI_PLATYPUS_CONTROL_H

#include <mpi.h>
#include <iosfwd>
#include <list>
#include <queue>
#include <portablethreads/time.h>
#include <platypus/answer_set_printer_base.h>
#include <platypus/types/partial_assignment.h>
#include <platypus/types/choice.h>
#include <distribution/mpi/mpi_platypus_types.h>
#include <distribution/mpi/mpi_serializer.h>
#include <distribution/mpi/mpi_statistics.h>
#include <distribution/mpi/mpi_control_state_handler.h>
#include <distribution/mpi/mpi_control_exit_handler.h>
#include <distribution/mpi/mpi_control_model_handler.h>

namespace Platypus
{
  class PartialAssignment;
  class MPISerializer;
  class AnswerSetPrinterBase;
  class ProgramInterface;
  class MPIControlAssistant;
  class MPIControlExitHandler;
  class MPIControlModelHandler;

  /**
   * Controls the behaviour of the master process within the cluster.
   *
   */
  class MPIControl
  {
  public:
    ~MPIControl();
    MPIControl(const ProgramInterface& program,
	       MPIStatistics& statistics,
	       AnswerSetPrinterBase& printer, 
	       size_t requested_answersets,
	       bool suppressPrinting);
    

    /**
     * Perfofrms the setup operations necesary to initilaize the 
     * master process.
     * 
     */
    void setup();

    /**
     * The loop controlling the master behaviour with respect to
     * messages sent to and received from workers.
     *
     */
    void start();

    /**
     * Returns the statistics gathered by the master from all the
     * worker processes.
     *
     */
    void finish(std::vector<unsigned long> & stats);
    
  private:
    MPIControl();

    /**
     * Tests to see whether the stop condition has been satisfied.
     * Presently this means that either all workers have requested and
     * been denied a choice from the master or the requested number
     * of answer sets have been found.
     */ 
    bool stop();

    /**
     * Test if enough answer sets have been received from the master.
     *
     */
    bool enoughReceived() const;

    /**
     * Print the answer set.
     *
     */
    void print(PartialAssignment& pa);

    /**
     * Send the termination messages to the workers.
     *
     */
    void initiateTermination();

    /**
     * receive termination confirmations from the workers.
     *
     */
    void completeTermination();


    /*
      void incMessSent(unsigned long inc = 1){ messSent_ += inc; }
      void incMessRecv(unsigned long inc = 1){ messRecv_ += inc; }
      unsigned long messSent() const { return messSent_; }
      unsigned long messRecv() const { return messRecv_; }
    */

    /**
     * Clean up any stray messages that may not have been received.
     */
    void cleanup();
    
  private:
    //void setUpPrequests();
    void requestDCFromDelegatedWorkers(unsigned source);
    void handleDelegatableChoice(unsigned source, unsigned count);
    void handleDelegatableChoiceRequest(unsigned source);
    void handleAnswerSet(unsigned source, unsigned count);
    void finish();

  private:
    const ProgramInterface* program_;
    MPISerializer* serializer_;
    MPIStatistics* statistics_;
    AnswerSetPrinterBase* printer_;
    MPIControlStateHandler* stateHandler_;
    MPIControlExitHandler* exitHandler_;
    MPIControlModelHandler* modelHandler_;
    PartialAssignment* pa_;
    unsigned requestedAnswerSets_;
    unsigned receivedAnswerSets_;
    unsigned globalAnswersFound_;
    bool suppressed_;
    unsigned requests_;
    bool shutdown_;
    unsigned workers_;

    MessageBuffer paBuffer_;
    MessageBuffer dcBuffer_;
    
    std::vector<unsigned> states_;
    std::queue<DelegatableChoice*> choices_;
    std::queue<unsigned> filedWorkers_;

    MPI::Status localControllerStatus_;
    MPI::Request localControllerRequest_;

    unsigned long answersFoundByWorker_;
    double start_time_;
  };
}

#endif
