/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <iostream>
#include <string>
#include <cstring>
#include <sstream>
#include <stdexcept>
#include "directory.h"
#include "program_options.h"
#include "value.h"
#include "system.h"
#include "path.h"
#include "string_functions.h"
#include "lowercase.h"
#include "build_flags.h"
#include "config.h"

using namespace std;
using namespace ProgramOptions;

string lib(lib_path()), include(include_path());

system_information_t system_description;
const Aliases aliases;
BuildFlags buildFlags;


bool contains(const std::string& haystack, const char* needle);
bool contains(const char* haystack, const char* needle);
void setPaths(const OptionValues& values);
bool setSystem(const OptionValues& values, system_information_t& info);

void printLibs(bool own, bool shared);
void completeSystemInformation(system_information_t& info);

string assembleGNUBuildStringFromSystemInformation()
{
	ostringstream s;
	s << system_description.machine_ << "-*-" << system_description.os_;
	return s.str();
}

int main(int argc, char** argv)
{
	OptionGroup options("This is the PortableThreads configuration utility.\nIt provides command line options to compile and link with PortableThreads.");
	options.addOptions()
		("help,h", bool_switch()->defaultValue(false), "print this")
		("defines", bool_switch()->defaultValue(false), "print defines to compile")
		("cxxflags", bool_switch()->defaultValue(false), "print flags for g++ to compile")
		("ldflags", bool_switch()->defaultValue(false), "print flags for g++ to link")
		("libs", bool_switch()->defaultValue(false), "print libs to link with")
		("lib-dir", value<string>(), "override default path for libraries")
		("include-dir", value<string>(), "override default path for header files")
		("no-own-libs", bool_switch()->defaultValue(false), "just put non-PortableThreads libraries with --libs")
		("shared", bool_switch()->defaultValue(false), "use shared libraries")
		("mode", value<string>()->defaultValue("detect"), "use 32 or 64 bit mode, default: detect")
		("system", value<string>(), "Provide flags for system (GNU build-string)")
		;

	try
	{
		OptionValues values(parseCommandLine(argc, argv, options, true));
		if(option_as<bool>(values, "help"))
		{
			cout << options << endl;
			return 0;
		}

		
		const string mode = option_as<string>(values, "mode");
		bool detect = mode == "detect";
		buildFlags.setDetectWidth(detect);
		
		if(!detect)
		{
			if(mode == "64")
			{
				buildFlags.setWidth(64);
			}
			else
			{
				buildFlags.setWidth(32);
			}
		}
		
		// via uname and their ilk
		getSystemInformation(system_description);
		string GNUBuildString = assembleGNUBuildStringFromSystemInformation();
		if(values.count("system") > 0)
		{
			GNUBuildString = option_as<string>(values, "system");
		}
		buildFlags.setGNUBuildString(GNUBuildString);
		
		setPaths(values);

		if(buildFlags.isConfigurationSupported())
		{

			if(option_as<bool>(values, "defines"))
			{
				cout << buildFlags.getDefines() << " "; 
			}
			if(option_as<bool>(values, "cxxflags"))
			{
				// header directory
				cout << "-I" << include << " ";
				cout << buildFlags.getCXXFlags() << " "; 
			}
			if(option_as<bool>(values, "ldflags"))
			{
				cout << buildFlags.getLDFlags() << " ";
			}

			if(option_as<bool>(values, "libs"))
				printLibs(!option_as<bool>(values, "no-own-libs"), option_as<bool>(values, "shared"));

			cout << "\n";
		}
		else
		{
			cout << "unsupported configuration: '" << GNUBuildString << "'\n";
		}
	}
	catch(const exception& e)
	{
		cout << e.what() << endl;
		return 1;
	}
	catch(...)
	{
		cout << "unknown error, exiting" << endl;
		return 1;
	}

	return 0;
}




void printLibs(bool own, bool shared)
{
	if(own)
	{
		// when using shared lib print -L<libdir> -lportablethreads,
		// otherwise print <libdir>/libportablethreads.a
		if(shared)
		{
			if(buildFlags.getOS() == Aliases::MINGW32)
			{
				cout << lib << '/' << "libportablethreads.dll.a ";
			}
			else
			{
				cout << "-L" << lib << " -lportablethreads ";
			}
		}
		else
		{
			cout << lib << '/' << "libportablethreads.a ";
		}
	}

	cout << buildFlags.getLibraries() << " ";
}



void setPaths(const OptionValues& values)
{
	if(	values.count("lib-dir") )
	{	
		lib = option_as<string>(values, "lib-dir");
		if(!directoryExists(lib))
			throw std::runtime_error("value for option --lib-dir must be a directory!");
	}
	if(	values.count("include-dir") )
	{	
		include = option_as<string>(values, "include-dir");
		if(!directoryExists(include))
			throw std::runtime_error("value for option --include-dir must be a directory!");
	}	
}

bool contains(const std::string& haystack, const char* needle)
{
	return haystack.find(needle) != std::string::npos;
}

bool contains(const char* haystack, const char* needle)
{
	return strstr(haystack, needle) != NULL;
}



