/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef ARCH_COMMON_H
#define ARCH_COMMON_H

#include <cstddef>

#define PT_ARCH_COMMON_INCLUDED

namespace PortableThreads 
{
	// Interger type definitions based on size
	namespace Private
	{
		template<unsigned>
		struct PTSizeToType;

		template<>
		struct PTSizeToType<1>
		{
			typedef int8 int_t;
			typedef uint8 uint_t;
		};

		template<>
		struct PTSizeToType<2>
		{
			typedef int16 int_t;
			typedef uint16 uint_t;
		};

		template<>
		struct PTSizeToType<4>
		{
			typedef int32 int_t;
			typedef uint32 uint_t;
		};

		template<>
		struct PTSizeToType<8>
		{
			typedef int64 int_t;
			typedef uint64 uint_t;
		};

		template<typename T>
		struct PTSign;

		template<>
		struct PTSign<int8>
		{
			enum __dummy {sign = 1};
		};
		template<>
		struct PTSign<int16>
		{
			enum __dummy {sign = 1};
		};
		template<>
		struct PTSign<int32>
		{
			enum __dummy {sign = 1};
		};
		template<>
		struct PTSign<int64>
		{
			enum __dummy {sign = 1};
		};
		template<>
		struct PTSign<uint8>
		{
			enum __dummy {sign = 0};
		};
		template<>
		struct PTSign<uint16>
		{
			enum __dummy {sign = 0};
		};
		template<>
		struct PTSign<uint32>
		{
			enum __dummy {sign = 0};
		};
		template<>
		struct PTSign<uint64>
		{
			enum __dummy {sign = 0};
		};

		template<typename T>
		struct PTCompilerHelper
		{
			typedef typename PTSizeToType<sizeof(T)>::uint_t uint_t;
			typedef typename PTSizeToType<sizeof(T)>::int_t int_t;
		};

		template<std::size_t N>
		struct PTSizeInDoubles
		{
			enum
			{
				NR_DOUBLES = N / sizeof(double) + ((N % sizeof(double)) != 0)
			};
		};

		template<typename T>
		struct PTAlignedSize
		{
			enum
			{
				SIZE = PTSizeInDoubles<sizeof(T)>::NR_DOUBLES * sizeof(double)
			};
		};
	}

	typedef Private::PTSizeToType<sizeof(void*)>::int_t pt_pointer_t;
	typedef pt_pointer_t pt_pointer_type; // backward compatibility
	typedef Private::PTSizeToType<sizeof(long)>::int_t pt_int_t;
	typedef pt_int_t pt_int_type; // backward compatibility
	typedef Private::PTSizeToType<sizeof(unsigned long)>::uint_t pt_uint_t;
	typedef pt_uint_t pt_uint_type; // backward compatibility
	

	namespace LockFree
	{
		namespace Private
		{
			template<bool HAS_COUNTER, bool HAS_USER_BITS>
			class token_t
			{
			public:	
				typedef pt_pointer_t int_t;
				explicit
				token_t(int_t pointer = 0, int_t count = 0, int_t userbits = 0)
					:	pointer_(pointer)
					,	count_(count)
					,	userbits_(userbits)
				{}
				inline int_t pointer() const { return pointer_; }
				inline void pointer(int_t v) { pointer_ = v; }
				inline int_t count() const { return count_; }
				inline void count(int_t c) { count_ = c; }
				inline int_t userbits() const { return userbits_; }
				inline void userbits(int_t c) { userbits_ = c; }
			private:
				int_t pointer_, count_, userbits_;
			};

			template<>
			class token_t<false, true>
			{
			public:	
				typedef pt_pointer_t int_t;
				explicit
				token_t(int_t pointer = 0, int_t = 0, int_t userbits = 0)
					:	pointer_(pointer)
					,	userbits_(userbits)
				{}
				inline int_t pointer() const { return pointer_; }
				inline void pointer(int_t v) { pointer_ = v; }
				inline int_t count() const { return 0; }
				inline void count(int_t) {}
				inline int_t userbits() const { return userbits_; }
				inline void userbits(int_t c) { userbits_ = c; }
			private:
				int_t pointer_, userbits_;
			};

			template<>
			class token_t<true, false>
			{
			public:	
				typedef pt_pointer_t int_t;
				explicit
				token_t(int_t pointer = 0, int_t count = 0, int_t = 0)
					:	pointer_(pointer)
					,	count_(count)
				{}
				inline int_t pointer() const { return pointer_; }
				inline void pointer(int_t v) { pointer_ = v; }
				inline int_t count() const { return count_; }
				inline void count(int_t c) { count_ = c; }
				inline int_t userbits() const { return 0; }
				inline void userbits(int_t) {}
			private:
				int_t pointer_, count_;
			};

			template<>
			class token_t<false, false>
			{
			public:	
				typedef pt_pointer_t int_t;
				explicit
				token_t(int_t pointer = 0, int_t = 0, int_t = 0)
					:	pointer_(pointer)
				{}
				inline int_t pointer() const { return pointer_; }
				inline void pointer(int_t v) { pointer_ = v; }
				inline int_t count() const { return 0; }
				inline void count(int_t) {}
				inline int_t userbits() const { return 0; }
				inline void userbits(int_t) {}
			private:
				int_t pointer_;
			};
			
			template<bool HAS_COUNTER, bool HAS_USER_BITS>
			inline bool operator==(	const token_t<HAS_COUNTER, HAS_USER_BITS>& lhs, 
									const token_t<HAS_COUNTER, HAS_USER_BITS>& rhs)
			{
				return	lhs.pointer() == rhs.pointer() && 
						lhs.count() == rhs.count() && 
						lhs.userbits() == rhs.userbits();
			}

			template<bool HAS_COUNTER, bool HAS_USER_BITS>
			inline bool operator!=(	const token_t<HAS_COUNTER, HAS_USER_BITS>& lhs, 
									const token_t<HAS_COUNTER, HAS_USER_BITS>& rhs)
			{
				return !(lhs == rhs);
			}
		}
	}
}



#endif
