/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef IA_64_GCC_H
#define IA_64_GCC_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

#define PT_HAVE_CAS

#include <cassert>
#include <portablethreads/arch/64-gcc.h>
#include <portablethreads/arch/no-tick-register.h>
#include <portablethreads/arch/cas-based-64-bit-atomic-add.h>
#include <portablethreads/arch/cas-based-64-bit-atomic-set.h>

/*
#define ia64_cmpxchg4_acq(ptr, new, old)						\
({											\
	__u64 ia64_intri_res;								\
	asm volatile ("mov ar.ccv=%0;;" :: "rO"(old));					\
	asm volatile ("cmpxchg4.acq %0=[%1],%2,ar.ccv":					\
			      "=r"(ia64_intri_res) : "r"(ptr), "r"(new) : "memory");	\
	ia64_intri_res;									\
})

#define ia64_cmpxchg8_acq(ptr, new, old)						\
({											\
	__u64 ia64_intri_res;								\
	asm volatile ("mov ar.ccv=%0;;" :: "rO"(old));					\
	asm volatile ("cmpxchg8.acq %0=[%1],%2,ar.ccv":					\
			      "=r"(ia64_intri_res) : "r"(ptr), "r"(new) : "memory");	\
	ia64_intri_res;									\
})

#define ia64_cmpxchg1_acq(ptr, new, old)						\
({											\
	__u64 ia64_intri_res;								\
	asm volatile ("mov ar.ccv=%0;;" :: "rO"(old));					\
	asm volatile ("cmpxchg1.acq %0=[%1],%2,ar.ccv":					\
			      "=r"(ia64_intri_res) : "r"(ptr), "r"(new) : "memory");	\
	ia64_intri_res;									\
})
*/

namespace PortableThreads 
{
	// Most of these functions are due to the Linux kernel.
	namespace LockFree
	{
		namespace Private
		{
			typedef uint8 spinlock_t;
			inline void pt_atomic_clear_lock(volatile spinlock_t* lock)
			{
				uint64 prev;	
				__asm__ __volatile__("mov ar.ccv=%0;;\t\n" :: "rO"(0xff));
				__asm__ __volatile__(
					"cmpxchg1.acq %0=[%1],%2,ar.ccv\t\n"
					: "=r"(prev) 
					: "r"(lock), "r"(0) 
					: "memory"
				);
			}
			
			inline bool pt_atomic_set_lock(volatile spinlock_t* lock)
			{
				uint64 prev;	
				__asm__ __volatile__("mov ar.ccv=%0;;\t\n" :: "rO"(0));
				__asm__ __volatile__(
					"cmpxchg1.acq %0=[%1],%2,ar.ccv\t\n"
					: "=r"(prev) 
					: "r"(lock), "r"(0xff) 
					: "memory"
				);
				return static_cast<uint8>(prev) == 0;
			}
			
			inline int32 pt_atomic_cas_return_memory(volatile int32* mem, int32 nv, int32 ov)
			{
				int64 prev;	
				__asm__ __volatile__("mov ar.ccv=%0;;\t\n" :: "rO"(ov));
				__asm__ __volatile__(
					"cmpxchg4.acq %0=[%1],%2,ar.ccv"
					: "=r"(prev) 
					: "r"(mem), "r"(nv) 
					: "memory"
				);
				return static_cast<int32>(prev);
			}
			
			inline int64 pt_atomic_cas_return_memory(volatile int64* mem, int64 nv, int64 ov)
			{
				int64 prev;	
				__asm__ __volatile__("mov ar.ccv=%0;;\t\n" :: "rO"(ov));
				__asm__ __volatile__(
					"cmpxchg8.acq %0=[%1],%2,ar.ccv"
					: "=r"(prev) 
					: "r"(mem), "r"(nv) 
					: "memory"
				);
				return prev;
			}
			
			inline bool pt_atomic_cas(volatile int32* mem, int32 nv, int32 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}
			inline bool pt_atomic_cas(volatile int64* mem, int64 nv, int64 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}

			inline void pt_mfence()
			{
				__asm__ __volatile__("mf" ::: "memory");
			}

			
			const unsigned HARDWARE_POINTER_BITS = 50; // Intel Itanium2 Programming Manual
			const unsigned ALIGNMENT_BITS = 3;
		}
	}	
}

#include <portablethreads/arch/arch-common.h>
#include <portablethreads/arch/native-atomic-arithmetic-common.h>
#include <portablethreads/arch/native-pointer-cas.h>
#include <portablethreads/arch/native-atomic-number.h>
#include <portablethreads/arch/ia-64-muxer.h>
#include <portablethreads/arch/cas-based-pointer-llsc.h>


namespace PortableThreads
{
	namespace LockFree
	{
		namespace Private
		{
			template<unsigned USER_BITS, unsigned ALIGNMENT = ALIGNMENT_BITS>
			struct PTPointerCASType
			{
				typedef PointerCAS< IA64Muxer<int64, HARDWARE_POINTER_BITS, ALIGNMENT, USER_BITS> > PTPointerCAS;
			};

			template<unsigned USER_BITS, unsigned ALIGNMENT = ALIGNMENT_BITS>
			struct PTPointerLLSCType
			{
				typedef PointerLLSC< IA64Muxer<int64, HARDWARE_POINTER_BITS, ALIGNMENT, USER_BITS> > PointerLLSC;
			};

			typedef PTPointerLLSCType<0>::PointerLLSC PTPointerLLSC;
		}
	}
}


#endif
