/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef SPARC_V8PLUS_GCC_COMMON_H
#define SPARC_V8PLUS_GCC_COMMON_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

#include <portablethreads/arch/sparc-v8-gcc-common.h>
#include <portablethreads/arch/cas-based-64-bit-atomic-add.h>
#include <portablethreads/arch/cas-based-64-bit-atomic-set.h>

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{

			// The SPARC V8+ architecture/abi is a sun-specific thing which has some of the V9's 
			// instructions as well as 64-bit register support.
		
			// prototypes
			inline int64 pt_atomic_cas_return_memory(volatile int64* mem, int64 nv, int64 ov);
			inline uint64 pt_ticks();
	
			/*
			* Atomic compare and exchange.  Compare OLD with MEM, if identical,
			* store NEW in MEM.  Return the initial value in MEM.  Success is
			* indicated by comparing RETURN with OLD.
			*/
			inline int32 pt_atomic_cas_return_memory(volatile int32* mem, int32 nv, int32 ov)
			{
				__asm__ __volatile__("cas [%2], %3, %0 \n\t"
					"membar #StoreLoad | #StoreStore \n\t"
					: "=&r" (nv)
					: "0" (nv), "r" (mem), "r" (ov)
					: "memory");

				return nv;
			}

			inline bool pt_atomic_cas(volatile int32* mem, int32 nv, int32 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}


			inline void pt_mfence()
			{
				__asm__ __volatile__("membar #LoadLoad | #LoadStore | #StoreStore | #StoreLoad \n\t" : : : "memory");
			}


			// atomically adds value to counter and returns the result
			inline int32 pt_atomic_add(volatile int32* counter, int32 value)
			{
				int32 current;
				do { current = *counter; } while(!pt_atomic_cas(counter, current+value, current));
				return current + value;
			}

			inline bool pt_atomic_cas(volatile int64* mem, int64 nv, int64 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}			
		}
	}

	inline uint64 pt_seed()
	{
		return LockFree::Private::pt_ticks();
	}
}


#endif
