/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
 
#ifndef SPINLOCK_ATOMIC_NUMBER_H
#define SPINLOCK_ATOMIC_NUMBER_H

#ifndef PT_ARCH_COMMON_INCLUDED
#	error "Architectural types and functions must be included first!"
#endif 

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace ArchSpecific
		{
			template<unsigned BITS>
			class PTAtomicNumber
			{
			public:
				typedef typename PortableThreads::Private::PTSizeToType<BITS / 8>::int_t int_t;
				// Constructs an object with an initial value of v
				PTAtomicNumber(int_t v = 0)
					:	value_(v)
					,	lock_(0)
				{}
				PTAtomicNumber(const PTAtomicNumber& other)
					:	value_(other.value_)
					,	lock_(0)
				{}
				PTAtomicNumber& operator=(const PTAtomicNumber& other)
				{
					value_ = other.value_;
					return *this;
				}
				inline int_t dec(int_t value = 1)
				{
					lock();
					value_ -= value;
					Private::pt_mfence();
					const int_t ret = value_;
					unlock();
					return ret;
				}
				inline int_t inc(int_t value = 1)
				{
					lock();
					value_ += value;
					Private::pt_mfence();
					const int_t ret = value_;
					unlock();
					return ret;
				}
				inline int_t get() const
				{
					return value_;
				}
				inline bool cas(int_t newValue, int_t oldValue)
				{
					lock();
					const bool same = value_ == oldValue;
					if(same)
					{
						value_ = newValue;
						Private::pt_mfence();						
					}
					unlock();
					return same;
				}
				inline void set(int_t v)
				{
					Private::pt_atomic_set(&value_, v);
				}
			private:
				inline void lock()
				{
					while(true) { if(Private::pt_atomic_set_lock(&lock_)) break; }
				}
				inline void unlock()
				{
					Private::pt_atomic_clear_lock(&lock_);
				}
			private:
				volatile int_t value_;
//				volatile spinlock_t lock_;
				volatile Private::ZZspinlock_t lock_;
			};
		}
	}
}

#endif
