/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef X86_32_GCC_H
#define X86_32_GCC_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

#define PT_HAVE_CAS
#define PT_HAVE_CAS2

#include <portablethreads/arch/32-gcc.h>
#include <portablethreads/arch/x86-32-gcc-common.h>
#include <portablethreads/arch/cas-based-64-bit-atomic-add.h>
#include <portablethreads/arch/cas-based-64-bit-atomic-set.h>

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
			// 64 bit
			inline bool pt_atomic_cas(volatile int64* mem, int64 newValue, int64 oldValue)
			{
				/* (from Pentium Manual)
				Compares the 64-bit value in EDX:EAX with the operand (destination operand). If the values
				are equal, the 64-bit value in ECX:EBX is stored in the destination operand. Otherwise, the
				value in the destination operand is loaded into EDX:EAX. The destination operand is an 8-byte
				memory location. For the EDX:EAX and ECX:EBX register pairs, EDX and ECX contain the
				high-order 32 bits and EAX and EBX contain the low-order 32 bits of a 64-bit value.
				*/

				// NOTE: This code is only so complicated b/c in PIC mode we cannot
				// use the ebx register, not even in the clobber list. However, b/c
				// cmpxchg8b REQUIRES the ebx register we need to store its content
				// first, do the op and then restore ebx to its previous content.

				int32 buf[] = { static_cast<int32>(newValue), 0 };

				__asm__ __volatile__(
					"push %%ebx				\n\t"
					"movl 0(%0), %%ebx		\n\t"
					"lock; cmpxchg8b (%1)   \n\t"
					"sete 4(%0)				\n\t"
					"pop %%ebx				\n\t"
					:
					:	"S"(buf), "D"(mem), 
						"a"(static_cast<int32>(oldValue)), 
						"d"(static_cast<int32>(oldValue >> 32)), 
						"c"(static_cast<int32>(newValue >> 32))
					: "cc", "memory"
				);

				return buf[1] != 0;
			}

			const unsigned HARDWARE_POINTER_BITS = 32;
			const unsigned ALIGNMENT_BITS = 2;
		}
	}
}

#include <portablethreads/arch/arch-common.h>
#include <portablethreads/arch/native-atomic-arithmetic-common.h>
#include <portablethreads/arch/native-pointer-cas.h>
#include <portablethreads/arch/native-atomic-number.h>
#include <portablethreads/arch/free-high-bits-muxer.h>
#include <portablethreads/arch/cas-based-pointer-llsc.h>


namespace PortableThreads
{
	namespace LockFree
	{
		namespace Private
		{
			template<unsigned USER_BITS, unsigned ALIGNMENT = ALIGNMENT_BITS>
			struct PTPointerCASType
			{
				typedef PointerCAS< FreeHighBitsMuxer<int64, 64, HARDWARE_POINTER_BITS, ALIGNMENT, USER_BITS> > PTPointerCAS;
			};

			typedef PTPointerCASType<0>::PTPointerCAS PTPointerCAS;

			template<unsigned USER_BITS, unsigned ALIGNMENT = ALIGNMENT_BITS>
			struct PTPointerLLSCType
			{
				typedef PointerLLSC< FreeHighBitsMuxer<int64, 64, HARDWARE_POINTER_BITS, ALIGNMENT, USER_BITS> > PointerLLSC;
			};

			typedef PTPointerLLSCType<0>::PointerLLSC PTPointerLLSC;
		}
	}
}

#endif
