/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef X86_64_GCC_H
#define X86_64_GCC_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

#define PT_HAVE_CAS

#include <cassert>
#include <portablethreads/arch/64-gcc.h>
#include <portablethreads/arch/x86-32-gcc-common.h>


namespace PortableThreads 
{
	// Most of these functions are due to the Linux kernel.
	namespace LockFree
	{
		namespace Private
		{
			inline int64 pt_atomic_add(volatile int64* counter, int64 value)
			{
				const int64 res = value;
				__asm__ __volatile__(
					"lock; xaddq %0, %1"
					:"=r"(value)
					:"m"(*counter), "0"(value)
				);
				return res + value;
			}

			/*
			* Atomic compare and exchange.  Compare OLD with MEM, if identical,
			* store NEW in MEM.  Return the initial value in MEM.  Success is
			* indicated by comparing RETURN with OLD.
			*/
			inline int64 pt_atomic_cas_return_memory(volatile int64* inMemory, int64 newValue, int64 oldValue)
			{
				int64 prev;	
				__asm__ __volatile__(
					"lock; cmpxchgq %1, %2"
					: "=a"(prev)
					: "q"(newValue), "m"(*inMemory), "0"(oldValue)
					: "memory"
				);
				return prev;
			}

			inline bool pt_atomic_cas(volatile int64* mem, int64 nv, int64 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}

			inline int64 pt_atomic_set(volatile int64* inMemory, int64 newValue)
			{
				__asm__ __volatile__(
					"xchgq %0, %1"
					:"=r" (newValue)
					:"m" (*inMemory), "0" (newValue)
					:"memory"
				);
				return newValue;
			}

			const unsigned HARDWARE_POINTER_BITS = 48;
			const unsigned ALIGNMENT_BITS = 3;
		}
	}
}

#include <portablethreads/arch/arch-common.h>
#include <portablethreads/arch/native-atomic-arithmetic-common.h>
#include <portablethreads/arch/native-pointer-cas.h>
#include <portablethreads/arch/native-atomic-number.h>
#include <portablethreads/arch/free-high-bits-muxer.h>
#include <portablethreads/arch/cas-based-pointer-llsc.h>

namespace PortableThreads
{
	namespace LockFree
	{
		namespace Private
		{
			// AMD64 & Intel's EM64T:
			// Both currently use 48 bit for addressing. Hence we may
			// safely use most significant 16 bits as well as the least significant 3 bits
			// for reference counting. It is assumed that pointers only point to data alligned
			// on at least 8 byte boundaries.
			// 
			// NOTE: Intel's EM64T cpus support cmpxchg16b, AMD's cpus not (yet).
			// AMD64 and Intel EM64T define an address to be in cannonical form 
			// if it is either [0...0|pointer] or [1...1|pointer]. Where pointer is in 
			// both cases 48-bit wide.

			template<unsigned USER_BITS, unsigned ALIGNMENT = ALIGNMENT_BITS>
			struct PTPointerCASType
			{
				typedef PointerCAS< FreeHighBitsMuxer<int64, 64, HARDWARE_POINTER_BITS, ALIGNMENT, USER_BITS> > PTPointerCAS;
			};

			typedef PTPointerCASType<0>::PTPointerCAS PTPointerCAS;

			template<unsigned USER_BITS, unsigned ALIGNMENT = ALIGNMENT_BITS>
			struct PTPointerLLSCType
			{
				typedef PointerLLSC< FreeHighBitsMuxer<int64, 64, HARDWARE_POINTER_BITS, ALIGNMENT, USER_BITS> > PointerLLSC;
			};

			typedef PTPointerLLSCType<0>::PointerLLSC PTPointerLLSC;
		}
	}
}


#endif
