/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_CONDITION_H
#define PT_CONDITION_H
#include <portablethreads/config.h>
#include <portablethreads/semaphore.h>
#include <portablethreads/lockfree/atomic_number.h>

#include <portablethreads/warning_header.h>

namespace PortableThreads
{
	/*! \class PTCondition condition.h portablethreads/condition.h
		\brief Condition variable.
	*/
	class LIBPORTABLETHREADS_API PTCondition
	{
	public:
		~PTCondition();
		/*!	\brief Create a condition variable 

			\param autoreset If set to *true* the condition variable will reset
				itself to the non-signaled state after it has been signaled.
			\param signaled If set to *true* the condition variable is created 
				signaled.

			\exception PTParameterError A condition variable cannot be created autresetting and signaled.
		*/
		PTCondition(bool autoreset = true, bool signaled = false);
		/*! \brief Causes waiting threads to become unblocked.

			The invokation of the method releases all waiting threads that are currently
			blocking on the condition variable. N Concurrent invokations of the method
			may result in the condition variable to become signaled anywhere from 1 to N 
			times.
		*/
		void signal();
		/*! \brief Resets a signaled condition variable to the non-signaled state.

			If the condition variable is in the non-signaled state or the
			condition variable uses autoressting (see PTCondition(bool, bool))
			the invokation of this method has no effect.
		*/
		void reset();
		/*! \brief Wait for the condition variable to become signaled.

			The execution of the calling thread is suspended until signal()
			is called.
		*/
		void wait();
		/*! \brief Wait for the condition variable to become signaled.

			The execution of the calling thread is suspended for at period of 
			atmost 
				\a seconds * 1000 + \a milliseconds ms. 
			If during this time signal() is called at
			least once, the waiting thread is unblocked. 

			During timed waiting a thread is woken every
			\a wakeup ms to see whether the condition
			variable became signaled. Smaller values increase
			the accuracy of the timed wait but consume more
			resources. If the parameter is not zero the
			wait period is set to \wakeup . Otherwise
			the wait period remains unchanged. 
			The default value is 100.

			\param seconds Seconds to sleep.
			\param seconds Milliseconds to add to sleep.
			\param wakeup Interrupt wait every \a wakeup ms to check if the condition became signaled.

			\retval true The condition variable became signaled during the wait.
			\retval false The condition variable did not become signaled during the wait.
		*/
		bool wait(unsigned seconds, unsigned milliseconds = 0, unsigned wakeup = 100);
	private:
		PTCondition(const PTCondition&);
		PTCondition& operator=(const PTCondition&);
	private:
		typedef LockFree::PTAtomicNumber::int_t int_t;
		PTSemaphore sem_;
		LockFree::PTAtomicNumber waiters_;
		volatile int incarnation_;
		bool autoreset_;
	};
}

#include <portablethreads/warning_footer.h>

#endif
