/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_LOCK_FREE_ALLOCATOR_H
#define PT_LOCK_FREE_ALLOCATOR_H
#include <portablethreads/config.h>
#include <portablethreads/exception.h>
#include <portablethreads/lockfree/memory_heap.h>
#include <limits>
#include <cstddef>

// windows.h
#ifdef min
#	undef min
#endif
#ifdef max
#	undef max
#endif

namespace PortableThreads 
{
	namespace LockFree
	{
		/*! \class PTAllocator allocator.h portablethreads/lock_deallocate.h
			\brief STL-compatible allocator class which uses 
				PortableThreads' lock-deallocate heap
				PTHeap to acquire memory.

			See the std::allocator for documentation.
		*/
		template<typename T>
		class PTAllocator
		{
		public:
			typedef const T* const_pointer;
			typedef const T& const_reference;
			typedef std::ptrdiff_t difference_type;
			typedef T* pointer;
			typedef T& reference;
			typedef std::size_t size_type;
			typedef T value_type;
		public:
			explicit
			PTAllocator()
			{}
			template<class Other>
			PTAllocator(const PTAllocator<Other>& other)
			{}
			inline pointer address(reference r) const { return &r; }
			inline const_pointer address(const_reference r) const { return &r; }
			inline pointer allocate(size_type count)
			{
				static PTHeap* heap = &PTHeap::getProcessHeap();
				return static_cast<pointer>(heap->allocate(count*sizeof(T)));
			}
			template<class Other>
			inline pointer allocate(size_type count, const Other* = 0)
			{
				static PTHeap* heap = &PTHeap::getProcessHeap();
				return static_cast<pointer>(heap->allocate(count*sizeof(T)));
			}
			inline void construct(pointer p, const_reference r)
			{
				new (p) T(r);
			}
			inline void deallocate(pointer p, size_type)
			{
				static PTHeap* heap = &PTHeap::getProcessHeap();
				heap->deallocate(p);
			}
			inline void destroy(pointer p)
			{
				p->~T();
			}
			inline size_type max_size() const
			{
				return std::numeric_limits<size_type>::max() / 2;
			}
			template<class Other>
			struct rebind 
			{
				typedef PTAllocator<Other> other;
			};
			template<class Other>
			inline PTAllocator& operator=(const PTAllocator<Other>& other)
			{
				return *this;
			}
		};

		template<typename T>
		inline bool operator==(const PTAllocator<T>& lhs, const PTAllocator<T>& rhs)
		{
			return true;
		}

		template<typename T>
		inline bool operator!=(const PTAllocator<T>& lhs, const PTAllocator<T>& rhs)
		{
			return false;
		}


		/*! \class PTHeapMixin lock_deallocate_allocator.h portablethreads/lock_deallocate.h
			\brief Mixin class that provides memory management for a given class using
				PortableThreads' lock-deallocate heap implementation PTHeap.

		*/
		class PTHeapMixin
		{
		public:
			typedef std::size_t size_t;
			//! allocate \a s bytes of memory.
			inline void* operator new(size_t s)
			{
				static PTHeap* heap = &PTHeap::getProcessHeap();
				return heap->allocate(s);
			}
			//! deallocate \a s bytes of memory starting at address \a p.
			void operator delete(void* p, size_t s)
			{
				static PTHeap* heap = &PTHeap::getProcessHeap();
				return heap->deallocate(p);
			}
			//! allocate \a s bytes of memory.
			void* operator new[](size_t s)
			{
				static PTHeap* heap = &PTHeap::getProcessHeap();
				return heap->allocate(s);
			}
			//! deallocate \a s bytes of memory starting at address \a p.
			void operator delete[](void* p, size_t s)
			{
				static PTHeap* heap = &PTHeap::getProcessHeap();
				return heap->deallocate(p);
			}
		};
	}
}

//#include <portablethreads/warning_footer.h>

#endif

