/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_ATOMIC_NUMBER_H
#define PT_ATOMIC_NUMBER_H
#include <portablethreads/config.h>

#include <portablethreads/warning_header.h>

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{

			/*! \class PTAtomicNumber atomic_number.h portablethreads/lock_free.h
				\brief Atomic arithmetic operations on integers.

				PTAtomicNumber provides atomic (read thread-safe) lock-free
				manipulation of an integer value, the number. Atomic numbers
				are useful e.g. to keep track of how many threads have passed
				some piece of code or to implement thread-safe reference counting.

				Different from other number-representing classes PTAtomicNumber's 
				methods DO NOT return a reference to the object but the NEW VALUE
				of the counter after the invokation of a method. Because a
				PTAtomicNumber object may be used in the context of multiple threads,
				the value of the counter may already have changed by the time a thread
				examines the content of the counter via PTAtomicNumber::get().

				Unless otherwise stated all methods manipulate a PTAtomicNumber object
				atomically. Exceptions are constructors, assignment operators and the 
				destructor.

				PTAtomicNumber uses an integer type which is of the natural width for
				the architecture. Typedefs for atomic numbers for these widths exists
				as well:

				- 32 bit: PTAtomicNumber32
				- 64 bit: PTAtomicNumber64

			*/
			template<unsigned BITS>
			class PTAtomicNumber : private ArchSpecific::PTAtomicNumber<BITS>
			{
				typedef ArchSpecific::PTAtomicNumber<BITS> Base;
			public:
				typedef typename Base::int_t int_t; //!< The integer type depends on number of bits requested.
				//! Constructs an object with an initial value of v.
				PTAtomicNumber(int_t v = 0)
					:	Base(v)
				{}
				PTAtomicNumber(const PTAtomicNumber& other)
					:	Base(other)
				{}
				PTAtomicNumber& operator=(const PTAtomicNumber& other)
				{
					Base::operator=(other);
					return *this;
				}
				//! Decrement by \a value.
				inline int_t dec(int_t value = 1)
				{
					return Base::dec(value);
				}
				//! Increment by \a value.
				inline int_t inc(int_t value = 1)
				{
					return Base::inc(value);
				}
				
				//! Decrement by \a value.
				inline int_t operator-=(int_t value)
				{
					return dec(value);
				}
				//! Increment by \a value.
				inline int_t operator+=(int_t value)
				{
					return inc(value);
				}
				//! Decrement by 1 return, new value.
				inline int_t operator--()
				{
					return dec();
				}
				//! Decrement by 1 return, old value.
				inline int_t operator--(int)
				{
					return dec() + 1;
				}
				//! Increment by 1 return, new value.
				inline int_t operator++()
				{
					return inc();
				}
				//! Increment by 1 return, old value.
				inline int_t operator++(int)
				{
					return inc() - 1;
				}
				/*! \brief Get the value of the number.

					This method is NOT atomic.
				*/
				inline int_t get() const
				{
					return Base::get();
				}
				//! \brief Set the value of the number.				
				inline int_t set(int_t v) const
				{
					Base::set(v);
				}
				/*! \brief Atomically compare-and-swap the content of the number.

					Provided the old value of the number is know (\a oldValue)
					the method atomically sets the number to the value of
					parameter \a newValue.

					\retval true Exchange happened.
					\retval false Exchange didn't happen.
				*/
				inline bool cas(int_t newValue, int_t oldValue)
				{
					return Base::cas(newValue, oldValue);
				}
				//! \brief Atomically compare-and-swap the content of the number.
				inline bool cas(const PTAtomicNumber& newValue, const PTAtomicNumber& oldValue)
				{
					return Base::cas(newValue.get(), oldValue.get());
				}
#if 0
				/*! \brief Assign a value to the number.

					Even though the actual assigning takes place
					atomically, because a reference is returned
					the state of the object may have changed by
					the time a call to e.g. get() is executed.				
				*/
				inline PTAtomicNumber& operator=(int_t v)
				{
					Base::operator=(v);
					return *this;
				}
#endif
			};
		}
		//! Atomic number which is supported on all platforms. 
		typedef Private::PTAtomicNumber<sizeof(pt_int_t)*8> PTAtomicNumber;
		//! Atomic numbers of a given size may not be supported on all platforms. Use at your own risk.
		typedef Private::PTAtomicNumber<32> PTAtomicNumber32;
		typedef Private::PTAtomicNumber<64> PTAtomicNumber64;
	}
}


#include <portablethreads/warning_footer.h>

#endif

