/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_MEMORY_HEAP_H
#define PT_MEMORY_HEAP_H

#include <portablethreads/config.h>
#include <portablethreads/lockfree/memory_chunk_manager.h>
#include <cstddef>

namespace PortableThreads
{
	namespace LockFree
	{
		/*! \class PTHeap heap.h portablethreads/lock_free.h
		\brief Lock-free heap implementation inspired by Maged. M. Michael's
		lock-free heap implementation (http://www.research.ibm.com/people/m/michael/pldi-2004.pdf).		
		
		The heap consists of several small object heaps which increase in object size by sizeof(double),
		starting at 8 byte up to 2048 byte. Memory requests larger than 2048 byte are satisfied the os'
		memory mapping system.
		*/
		class LIBPORTABLETHREADS_API PTMemoryHeap
		{
		public:
			enum
			{
				MIN_CHUNK_SIZE = 8,
				MAX_CHUNK_SIZE = 2048,
				CHUNK_STEP = sizeof(double),
				NUMBER_OF_MANAGERS = MAX_CHUNK_SIZE / CHUNK_STEP
			};
			~PTMemoryHeap();
			PTMemoryHeap();
			//! Access to the singe process heap.
			static PTMemoryHeap& getProcessHeap();
			/*!	\brief Allocate memory
				
				\param fBytes Number of bytes to allocate.
				\exception Throws std::bad_alloc if no memory can be obtained from the system.
			*/			
			inline void* allocate(std::size_t fBytes)
			{
				return malloc(static_cast<pt_uint_t>(fBytes));
			}
			//! \brief Free previously allocated memory
			inline void deallocate(void* p)
			{
				free(p);
			}
			//! \brief Standard C function interface: malloc, reallow, free. These methods behave in accordance with the C89 standard.
			void* malloc(pt_uint_t Size);
			void* realloc(void* p, pt_uint_t Size);
			void free(void* p);
		private:
			void* copy(void* fSrc, pt_uint_t fOldSize, pt_uint_t fNewSize);
			PTMemoryHeap(PTMemoryHeap&);
			PTMemoryHeap& operator=(PTMemoryHeap&);
		private:
			enum
			{
				MANAGER_STORAGE_SIZE_IN_BYTES = PortableThreads::Private::PTAlignedSize<Private::PTMemoryChunkManager>::SIZE * NUMBER_OF_MANAGERS,
				MANAGER_STORAGE_SIZE_IN_DOUBLES = MANAGER_STORAGE_SIZE_IN_BYTES / sizeof(double) + (MANAGER_STORAGE_SIZE_IN_BYTES % sizeof(double) > 0)
			};
			double raw_[MANAGER_STORAGE_SIZE_IN_DOUBLES];
		};

		// backward compatability 
		typedef PTMemoryHeap PTHeap;
	}
}

#endif // PT_MEMORY_HEAP_H
