/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_LOCK_FREE_QUEUE_H
#define PT_LOCK_FREE_QUEUE_H
#include <portablethreads/config.h>
#include <portablethreads/lockfree/utility.h>
#include <cassert>
#include <iterator>
#include <memory> // std::allocator
#include <new> // placement new

#if	defined(_DEBUG) && defined(PT_WINDOWS)
#	include <windows.h>
#	undef assert
#	define assert(x) if(!(x)) DebugBreak()
#endif

#include <portablethreads/warning_header.h>

namespace PortableThreads 
{
	namespace LockFree
	{
		
		

		/*! \class PTQueue queue.h portablethreads/lock_free.h
			\brief  Lock-free queue with correct memory management.
		
			The queue implementation follows Michael and Scott's algorithm given
			in http://citeseer.ist.psu.edu/michael96simple.html

			Manipulation of the queue via the pushBack/popFront interface is thread-safe.

			When pushing an element into or removing an element from the queue,
			the element is ever copied or assigned within the context of 
			one thread.

			The queue allows for its content to be iterated over using
			forward iterators. The iterators may also be used to modify
			the content of the queue. Iterating over the queue yields
			undefined results if the stack is used concurrently.

			The allocator provided as template parameter must allow
			for concurrent allocation and deallocation. The STL allocator
			meets this requirement.

			Memory which is allocated by the queue implementation is
			freed on object destruction. 	
		*/
		template<typename T, typename A = std::allocator<T> >
		class PTQueue
		{
			typedef Private::PTPointerLLSC PTPointerLLSC;
			struct TNode
			{
				volatile char raw_[sizeof(T)];
				PTPointerLLSC next_;
			};
			struct HelperNode
			{
				PTPointerLLSC next_;
				volatile TNode* dataNode_;
			};
			struct Pair
			{
				HelperNode helperNode_;
				TNode tnode_;
			};			
		public:
			template<class U, class NodeType>
			class QueueIterator : public std::iterator<std::forward_iterator_tag, U>
			{
				typedef std::iterator<std::forward_iterator_tag, U> Base;
				typedef NodeType Node;
			public:
				typedef typename Base::pointer pointer;
				typedef typename Base::reference reference;
				operator const void*() const { return current_; }
				explicit
				QueueIterator(Node* node = 0)
					:	current_(node)
				{}
				inline pointer operator->() const
				{
					return &**this;
				}
				inline reference operator*() const
				{
					return *reinterpret_cast<U*>(const_cast<char*>(current_->dataNode_->raw_));
				}
				inline QueueIterator& operator++()
				{
					if(current_)
						current_ = reinterpret_cast<Node*>(current_->next_.get().pointer());
					return *this;
				}
				inline QueueIterator operator++(int)
				{
					const QueueIterator temp(*this);
					++(*this);
					return temp;
				}
			private:
				Node* current_;
			};
		public:
			typedef typename A::template rebind<Pair>::other allocator_type;
			typedef T value_type;
			typedef T& reference;
			typedef const T& const_reference;
			typedef QueueIterator<T, HelperNode> iterator;
			typedef QueueIterator<const T, HelperNode> const_iterator;
			~PTQueue()
			{
				// call dtors 
				for(HelperNode* h = reinterpret_cast<HelperNode*>(reinterpret_cast<HelperNode*>(head_.get().pointer())->next_.get().pointer());
					h ; h = reinterpret_cast<HelperNode*>(h->next_.get().pointer()))
				{
					reinterpret_cast<T*>(const_cast<char*>(h->dataNode_->raw_))->~T();
				}

				// free nodes (active queue)
				for(HelperNode* h = reinterpret_cast<HelperNode*>(head_.get().pointer());
					h ; )
				{
					HelperNode* n = reinterpret_cast<HelperNode*>(h->next_.get().pointer());
					Pair* p = reinterpret_cast<Pair*>(h);
					p->~Pair();
					allocator_.deallocate(p, 1);
					h = n;
				}

				// free nodes (recycle stack)
				for(HelperNode* h = reinterpret_cast<HelperNode*>(helperNodeStore_.get().pointer());
					h ; )
				{
					HelperNode* n = reinterpret_cast<HelperNode*>(h->next_.get().pointer());
					Pair* p = reinterpret_cast<Pair*>(h);
					p->~Pair();
					allocator_.deallocate(p, 1);
					h = n;
				}
			}
			//! Create an empty queue.
			explicit
			PTQueue()
				:	helperNodeStore_(0)
				,	tnodeStore_(0)
				,	head_(reinterpret_cast<typename PTPointerLLSC::token_t::int_t>(recycleHelperNode()))
				,	tail_(head_.get().pointer())
			{
				reinterpret_cast<HelperNode*>(head_.get().pointer())->dataNode_ = 0;
				Private::pt_mfence();
			}
			//! Create an empty queue using the allocator provided.
			explicit
			PTQueue(A& allocator)
				:	allocator_(allocator)
				,	helperNodeStore_(0)
				,	tnodeStore_(0)
				,	head_(reinterpret_cast<PTPointerLLSC::token_t::int_t>(recycleHelperNode()))
				,	tail_(head_.get().pointer())
			{
				reinterpret_cast<HelperNode*>(head_.get().pointer())->dataNode_ = 0;
				Private::pt_mfence();
			}
			/*! \brief Add one element to the back of the queue.

				\param value Element to add.

				\retval An iterator to the element in the queue.
			*/
			iterator pushBack(const_reference value)
			{
				TNode* tnode = recycleTNode();
				try
				{
					new (const_cast<char*>(tnode->raw_)) T(value);
				}
				catch(...)
				{
					recycleTNode(tnode);
					throw;
				}

				HelperNode* node = recycleHelperNode();
				// setup node
				{
					// set proper end
					PTPointerLLSC::token_t n = node->next_.get();
					n.pointer(0);
					node->next_.assign(n);

					node->dataNode_ = tnode;
				}

				Private::pt_queue_push_back(tail_, offsetof(HelperNode, next_), node);
				return iterator(node);
			}

			/*! \brief Remove the element at the front of the queue.

				\param value Reference which is assigned the element removed from the queue.

				\retval true An element was removed from the queue.
				\retval false The queue was empty at the time.
			*/
			bool popFront(reference value)
			{
				TNode* tnode = 0;
				HelperNode* recycleNode = 0;
				if(!Private::pt_queue_pop_front(head_, tail_, offsetof(HelperNode, next_), offsetof(HelperNode, dataNode_), recycleNode, tnode))
					return false;

				assert(recycleNode);				
				assert(tnode);
				
				//recycleHelperNode(reinterpret_cast<HelperNode*>(oh.pointer()));
				recycleHelperNode(recycleNode);

				assert(tnode);
				T* data = reinterpret_cast<T*>(const_cast<char*>(tnode->raw_));
				value = *data;
				data->~T();
				recycleTNode(tnode);
				
				return true;
			}
			//! Get an iterator to the frontmost element.
			inline iterator begin() 
			{ 
				return iterator(reinterpret_cast<HelperNode*>(
					reinterpret_cast<HelperNode*>(head_.get().pointer())->next_.get().pointer())); 
			}
			//! Get an iterator to the last element.
			inline iterator end() { return iterator(0); }
			//! Get an const_iterator to the frontmost element.
			inline const_iterator begin() const 
			{ 
				return const_iterator(reinterpret_cast<HelperNode*>(
					reinterpret_cast<HelperNode*>(head_.get().pointer())->next_.get().pointer())); 
			}
			//! Get an const_iterator to the last element.
			inline const_iterator end() const { return const_iterator(0); }
		private:
			HelperNode* recycleHelperNode()
			{
				HelperNode* node = static_cast<HelperNode*>(Private::pt_stack_pop(helperNodeStore_, offsetof(HelperNode, next_)));
				if(node)
					return node;

				Pair* pair = allocator_.allocate(1, static_cast<void*>(0));
				assert(pair);
				new (pair) Pair;
				recycleTNode(&pair->tnode_);
				return &pair->helperNode_;
			}
			inline void recycleHelperNode(HelperNode* node)
			{
				assert(node);
				Private::pt_stack_push(
					helperNodeStore_, 
					offsetof(HelperNode, next_), 
					node);
			}
			TNode* recycleTNode()
			{
				TNode* node = static_cast<TNode*>(Private::pt_stack_pop(tnodeStore_, offsetof(TNode, next_)));
				if(node)
					return node;
				
				Pair* pair = allocator_.allocate(1, static_cast<void*>(0));
				assert(pair);
				new (pair) Pair;
				recycleHelperNode(&pair->helperNode_);
				return &pair->tnode_;
			}
			inline void recycleTNode(TNode* node)
			{
				assert(node);
				Private::pt_stack_push(
					tnodeStore_, 
					offsetof(TNode, next_), 
					node);
			}
		private:
			allocator_type allocator_;
			PTPointerLLSC helperNodeStore_, tnodeStore_, head_, tail_;
		};
	}
}

#include <portablethreads/warning_footer.h>

#endif

