/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_LOCK_FREE_STACK_H
#define PT_LOCK_FREE_STACK_H
#include <portablethreads/config.h>
#include <portablethreads/lockfree/utility.h>
#include <cassert>
#include <iterator>
#include <memory> // std::allocator
#include <new> // placement new

#include <portablethreads/warning_header.h>

namespace PortableThreads 
{

	namespace LockFree
	{
		/*! \class PTStack stack.h portablethreads/lock_free.h
			\brief Lock-free stack with correct memory management.

			This stack implementation is based on Treiber's original
			lock-free stack implementation.
			
			Reference: R. K. Treiber. Systems Programming: Coping with
			Parallelism. In RJ 5118, IBM Almaden Research
			Center, April 1986.

			Manipulation of the stack via the push/pop interface is thread-safe.

			When pushing an element onto or popping an element from the stack,
			the element is ever copied or assigned within the context of 
			one thread.

			The stack allows for its content to be iterated over using
			forward iterators. The iterators may also be used to modify
			the content of the stack. Iterating over the stack yields
			undefined results if the stack is used concurrently.

			Memory which is allocated by the stack implementation is
			freed on object destruction. 			
		*/
		template<class T, class A = std::allocator<T> >
		class PTStack
		{
			typedef Private::PTPointerLLSC PTPointerLLSC;
			struct Node
			{
				volatile char raw_[sizeof(T)];
				PTPointerLLSC next_;
			};
		public:
			template<typename U>
			class StackIterator : public std::iterator<std::forward_iterator_tag, U> 
			{
				typedef std::iterator<std::forward_iterator_tag, U> Base;
			public:
				typedef typename Base::pointer pointer;
				typedef typename Base::reference reference;
				operator const void*() const { return current_; }
				explicit
				StackIterator(Node* node = 0)
					:	current_(node)
				{}
				inline pointer operator->() const
				{
					&**this;
				}
				inline reference operator*() const
				{
					return *reinterpret_cast<T*>(const_cast<char*>(current_->raw_));
				}
				inline StackIterator& operator++()
				{
					if(current_)
						current_ = reinterpret_cast<Node*>(current_->next_.get().pointer());
					return *this;
				}
				inline StackIterator operator++(int)
				{
					const StackIterator temp(*this);
					++(*this);
					return temp;
				}
			private:
				Node* current_;
			};
		public:
			typedef typename A::template rebind<Node>::other allocator_type;		
			typedef T value_type;
			typedef T& reference;
			typedef StackIterator<T> iterator;
			typedef StackIterator<const T> const_iterator;
			typedef const T& const_reference;
			~PTStack()
			{
				for(Node* n = pop(head_); n; )
				{
					Node* next = pop(n->next_);
					reinterpret_cast<T*>(const_cast<char*>(n->raw_))->~T();
					n->~Node();
					allocator_.deallocate(n, 1);
					n = next;
				}
				for(Node* n = pop(storeHead_); n; )
				{
					Node* next = pop(n->next_);
					n->~Node();
					allocator_.deallocate(n, 1);
					n = next;
				}
			}
			//! Create an empty stack.
			explicit
			PTStack()
				:	head_(0)
				,	storeHead_(0)
			{}
			//! Create an empty stack using the allocator provided.
			explicit
			PTStack(allocator_type& a)
				:	allocator_(a)
				,	head_(0)
				,	storeHead_(0)
			{}
			/*! \brief Push one element on the stack.

				\param t Element to push onto the stack.

				\retval An iterator to the element on the stack.
			*/
			inline iterator push(const_reference t)
			{
				Node* node = recycle();
				try
				{
					new (const_cast<char*>(node->raw_)) T(t);
				}
				catch(...)
				{
					recycle(node);
					throw;
				}
				iterator ret(node);

				push(head_, node);

				return ret;
			}
			/*! \brief Pop topmost element from the stack.

				If the pop operation does not succeed, the reference \a t is
				left unchanged.

				\param t Reference which is assigned the element popped from the stack

				\retval true An element was removed from the stack.
				\retval false The stack was empty at the time.
			*/
			inline bool pop(reference fReference)
			{
				Node* node = pop(head_);
				if(node)
				{
					T* t = reinterpret_cast<T*>(const_cast<char*>(node->raw_));
					fReference = *t;
					t->~T();
					recycle(node);
					return true;
				}
				return false;
			}
			//! Get an iterator to the topmost stack element.
			inline iterator begin() 
			{ 
				return iterator(reinterpret_cast<Node*>(head_.get().pointer())); 
			}
			//! Get an iterator to the last stack element.
			inline iterator end() { return iterator(0); }
			//! Get an const_iterator to the topmost stack element.
			inline const_iterator begin() const 
			{ 
				return const_iterator(
					reinterpret_cast<const Node*>(head_.get().pointer())); 
			}
			//! Get an const_iterator to the last stack element.
			inline const_iterator end() const { return const_iterator(0); }
		private:
			inline Node* pop(PTPointerLLSC& fHead)
			{
				return static_cast<Node*>(Private::pt_stack_pop(fHead, offsetof(Node, next_)));
			}
			void push(PTPointerLLSC& fHead, Node* fNode)
			{
				assert(fNode);
				Private::pt_stack_push(fHead, offsetof(Node, next_), fNode);
			}
			inline Node* recycle()
			{
				Node* node = pop(storeHead_);
				if(node)
					return node;
				return new (allocator_.allocate(1, static_cast<void*>(0))) Node;
			}
			inline void recycle(Node* node)
			{
				push(storeHead_, node);
			}
		private:
			allocator_type allocator_;
			PTPointerLLSC head_, storeHead_;
		};

	}

}

#include <portablethreads/warning_footer.h>

#endif

