/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_SEMAPHORE_H
#define PT_SEMAPHORE_H
#include <portablethreads/config.h>
#ifdef PT_WINDOWS
#	include <portablethreads/win32/semaphore.h>
#endif
#ifdef PT_UNIX
#	include <portablethreads/unix/semaphore.h>
#endif

#include <portablethreads/warning_header.h>

namespace PortableThreads
{
	/*! \class PTSemaphore semaphore.h portablethreads/semaphore.h
		\brief Semaphore

		PTSemaphore implements a semaphore object. Threads that are waiting on the
		semaphore are suspended until the semaphore's count is greater than 0.

		The count value of a semaphore determines how many threads may call
		PTSemaphore::down() without blocking.
	*/	
	class PTSemaphore : private OSSpecific::PTSemaphore
	{
		typedef OSSpecific::PTSemaphore Base;
	public:
		/*! \brief Create a semaphore object.

			\param initially Initial value of the semaphore's count.

			\exception PTResourceError Thrown if no operating system semaphore can be allocated.
		*/
		PTSemaphore(pt_uint_t initially = 0)
			:	Base(initially)
		{}
		/*! \brief Attempt to decrement the semaphore's counter.

			\retval true The counter was greater than zero prior to the call.
			\retval false The counter was less or equal to zero prior to the call.
		*/
		inline bool tryDown()
  		{
			return Base::tryDown();
		}
		/*! \brief Decrement the semaphore's counter.

			The calling thread is suspended if the counter is zero or
			less prior to the called.
		*/
		inline void down()
  		{ 
    		Base::down();
		}
		/*! \brief Increment the semaphore's counter.

			If the counter was less than zero prior to the call one thread
			which has called down() is resumed.
		*/
		inline void up()
  		{
  			Base::up();
  		}
	};
}

#include <portablethreads/warning_footer.h>

#endif
