/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
 
#ifndef PT_SPINLOCK_H
#define PT_SPINLOCK_H
#include <portablethreads/config.h>
#include <portablethreads/time.h>

namespace PortableThreads 
{
	/*! \class PTSpinlock spinlock.h portablethreads/spinlock.h
		\brief Spinlock

		PTSpinlock implements a mutual exclusion object. Threads that are waiting to
		acquire the object are busy waiting on the object until they acquire it.
	*/
	class PTSpinlock
	{
	public:
		/*! \brief Create a spinlock object.

			\param locked If true, the spinlock is locked, unlocked otherwise.
		*/
		PTSpinlock(bool locked = false)
			:	lock_(0)
		{
			if(locked)
				lock();
		}
		/*! \brief Spin the lock until it is locked.

			Busy-wait for the spinlock to become available. During
			the wait the calling thread consumes massive amounts of
			CPU resources.
		*/
		inline void lock()
		{
			while(true) { if(LockFree::Private::pt_atomic_set_lock(&lock_)) break; else pt_yield(); }
		}
		/*! \brief Take one shot at acquiring the lock.

			\retval true Lock was acquired by the calling thread.
			\retval false Lock could not be acquired by the calling thread.
		*/
		inline bool tryLock()
		{
			return LockFree::Private::pt_atomic_set_lock(&lock_);
		}
		/*! \brief Unlock the spinlock.

			Calling this method will enable one other thread which is
			currently spinning on the lock (trying to lock it) to
			acquire the lock.
		*/
		inline void unlock()
		{
			LockFree::Private::pt_atomic_clear_lock(&lock_);
		}
	private:
		PTSpinlock(const PTSpinlock&);
		PTSpinlock& operator=(const PTSpinlock&);
	private:
		volatile LockFree::Private::spinlock_t lock_;
	};
}

//#include <portablethreads/warning_footer.h>

#endif

