/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_TIME_H
#define PT_TIME_H
#include <portablethreads/config.h>
#include <cassert>

#include <portablethreads/warning_header.h>

namespace PortableThreads 
{
	/*!	\brief Suspends the calling thread.
		\param time Seconds to suspend the calling thread.
	*/
	LIBPORTABLETHREADS_API void pt_second_sleep(unsigned time);
	/*!	\brief Suspends the calling thread.
		\param time Milliseconds to suspend the calling thread.
	*/
	LIBPORTABLETHREADS_API void pt_milli_sleep(unsigned time);
	/*!	\brief Suspends the calling thread.
		\param time Microseconds to suspend the calling thread.
	*/
	LIBPORTABLETHREADS_API void pt_micro_sleep(unsigned time);
	//! \brief Causes the calling thread to give up the CPU.
	LIBPORTABLETHREADS_API void pt_yield();

	
	/*! \class PTime time.h portablethreads/time.h 
		\brief Time stamps and measuring. 
	
		On UNIX systems this has (should have) micro second precision
		on WIN32 systems precision depends on the hardware but should be better or equivalent to
		UNIX precision.
		
		PTime also provides a stop clock interface (via start, stop, difference) and time stamps.
	*/
	class LIBPORTABLETHREADS_API PTime
	{
	public:
		typedef uint64 time_type; //!< unsigned integer type used for time measurement
		~PTime();
		PTime();
		//! Obtain the current time stamp.
		static time_type now();
		//! Get the frequency of the clock in Hz.
		static time_type frequency();
		//! Starts (and resets) the stopwatch
		void start();
		//! Stops the stopwatch
		void stop();
		/*! \brief Get the time elapsed between a call to start() and a call to stop().
		
			Divide this value by the return value of frequency()
			to get the time in seconds.
			NOTE: Do not use integer division here, the time measured may
			be well less than one second.
		*/
		time_type difference() const;
	private:
		static time_type calculateFrequency();
	private:
		time_type start_, end_;
	private:
		static const time_type frequency_;
	};
}

#include <portablethreads/warning_footer.h>

#endif
