/*
    Copyright (C) 2006 Jean Gressmann (jean@0x42.de)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
 
#ifndef POSIX_NAMED_SEMAPHORE_H
#define POSIX_NAMED_SEMAPHORE_H
#include <portablethreads/config.h>
#include <portablethreads/exception.h>
#include <errno.h>
#include <unistd.h>
#include <semaphore.h>
#include <cstring>
#include <limits>
#include <sstream>


namespace PortableThreads 
{
	namespace OSSpecific
	{
		class PTSemaphore
		{
		public:
			PTSemaphore(pt_uint_t initially = 0)
			{
				const pt_uint_t MAX = getMaxSemValue();
				sem_ = sem_open("", O_CREAT | O_EXCL | 0600, 
				if(result == SEM_FAILED)
				{
					std::ostringstream s;
					s << "[PTSemaphore] Could not create semaphore. Error: " << std::strerror(errno);
					throw PTResourceError(s.str());
				}
			}
		  	
			// WARNING: make sure there is no thread waiting on the semaphore
			// because doing so will result in undefined behavior when the
			// semaphore is destroyed
			~PTSemaphore()
			{
				// no while loop, no error defined due to interruption
				sem_close(sem_);
			}
			
			inline bool tryDown()
  			{
				// I don't really care why it failed...
				return sem_trywait(sem_) != -1;
			}
		    
			inline void down()
  			{ 
    			// I don't really care why it failed...
				while(sem_wait(sem_) == -1);
			}
		    
			inline void up()
  			{
  				// ok, only error as of POSIX is "invalid sem"
    			sem_post(sem_);
  			}		
		private:
			PTSemaphore(const PTSemaphore&);
  			PTSemaphore& operator=(const PTSemaphore&);
			inline static unsigned getMaxSemValue()
			{
				const long x = sysconf(_SC_SEM_VALUE_MAX);
				const unsigned long m = std::numeric_limits<unsigned>::max();
				if(x == -1)
					return m;
				
				return static_cast<unsigned long>(x) > m ? m : x;
			}
		private:
			sem_t* sem_;
		};
	}
}

#endif
