/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
 
#ifndef UNIX_POSIX_SEMAPHORE_H
#define UNIX_POSIX_SEMAPHORE_H
#include <portablethreads/config.h>
#include <portablethreads/exception.h>
#include <errno.h>
#include <unistd.h>
#include <semaphore.h>
#include <cstring>
#include <limits>
#include <sstream>


namespace PortableThreads 
{
	namespace OSSpecific
	{
		class PTSemaphore
		{
		public:
			PTSemaphore(pt_uint_t initially = 0)
			{
				const pt_uint_t MAX = getMaxSemValue();	
				const int result = sem_init(&sem_, 0 /* shared between processes */, initially > MAX ? MAX : initially);
				if(result == -1)
				{
					std::ostringstream s;
					s << "[PTSemaphore] Could not create semaphore. Error: " << std::strerror(errno);
					throw PTResourceError(s.str());
				}
			}
		  	
			// WARNING: make sure there is no thread waiting on the semaphore
			// because doing so will result in undefined behavior when the
			// semaphore is destroyed
			~PTSemaphore()
			{
				// no while loop, no error defined due to interruption
				sem_destroy(&sem_);
			}
			
			inline bool tryDown()
  			{
				// I don't really care why it failed...
				return sem_trywait(&sem_) != -1;
			}
		    
			inline void down()
  			{ 
    			// I don't really care why it failed...
				while(sem_wait(&sem_) == -1);
			}
		    
			inline void up()
  			{
  				// ok, only error as of POSIX is "invalid sem"
    			sem_post(&sem_);
  			}		
		private:
			PTSemaphore(const PTSemaphore&);
  			PTSemaphore& operator=(const PTSemaphore&);
			inline static unsigned getMaxSemValue()
			{
				const long x = sysconf(_SC_SEM_VALUE_MAX);
				const unsigned long m = std::numeric_limits<unsigned>::max();
				if(x == -1)
					return m;
				
				return static_cast<unsigned long>(x) > m ? m : x;
			}
		private:
			sem_t sem_;
		};
	}
}

#endif
