/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
 
#ifndef UNIX_THREAD_H
#define UNIX_THREAD_H
#include <portablethreads/config.h>
#include <portablethreads/exception.h>
#include <pthread.h>
#include <sched.h>
#include <cassert>

namespace PortableThreads 
{
	namespace OSSpecific
	{
		struct ThreadTraits
		{
			struct thread_t
			{
				volatile pthread_t thread_;
				void* arg_;
			};
			typedef pthread_t os_id_t;
			typedef void* (os_entry_function_t)(void*);
			static pthread_t create(os_entry_function_t* entry, void* arg)
			{
				pthread_attr_t attr;
				pthread_attr_init(&attr);
				pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);	
				pthread_attr_setschedpolicy(&attr, get_default_schedule_policy());

				/*
				From Single Unix Spec:
				There is no requirement on the implementation that the ID of
				the created thread be available before the newly created thread 
				starts executing. The calling thread can obtain the ID of the created
				thread through the return value of the pthread_create() function,
				and the newly created thread can obtain its ID by a call to pthread_self().
				*/

				// Hence we pass a the pthread_t structure so the
				// newly created thread can set it's id in case the (presumable)
				// main thread that created it doesn't return from this function
				// before the created thread needs to find out it's id!
				pthread_t t;
				const int result = pthread_create(&t, &attr, entry, arg);

				// destroy attributes before checking return value :)
				pthread_attr_destroy(&attr);
				
				if(result != 0)
					throw PTResourceError("[PThread] Could not create thread");
				
				return t;
			}
			inline static void create(thread_t& t, os_entry_function_t* entry, void* arg)
			{
				assert(t.arg_ == 0);
				t.arg_ = arg;
				t.thread_ = create(entry, &t);
			}
			inline static void kill(thread_t& t) { pthread_cancel(t.thread_); }
			inline static void initialize(thread_t& t) { t.arg_ = 0; }
			inline static os_id_t self() { return pthread_self(); }
			inline static void yield() { sched_yield(); }
			inline static void free_resource(thread_t& t) 
			{
				t.arg_ = 0;
			}
			inline static bool equal(os_id_t lhs, os_id_t rhs)
			{
				return pthread_equal(lhs, rhs) != 0;
			}
			static os_id_t thread_id(const thread_t& t) { return t.thread_; }
			inline static void join(thread_t& t) { pthread_join(t.thread_, 0); }
			inline static void detach(thread_t& t) { pthread_detach(t.thread_); }
			static int do_get_default_schedule_policy()
			{
				pthread_attr_t attr;
				int policy;
				
				pthread_attr_init(&attr);				
				pthread_attr_getschedpolicy(&attr, &policy);
				pthread_attr_destroy(&attr);
				return policy;
			}
			inline static int get_default_schedule_policy()
			{
				static const int policy = do_get_default_schedule_policy();
				return policy;
			}
			inline static int get_lowest_priority() { return sched_get_priority_min(get_default_schedule_policy()); }
			inline static int do_get_below_normal_priority()
			{
				const int l = get_lowest_priority();
				const int d = get_highest_priority() - l;
				return l + d / 4;
			}
			inline static int get_below_normal_priority() 
			{
				static const int priority = do_get_below_normal_priority();
				return priority; 
			}
			inline static int do_get_normal_priority()
			{
				const int l = get_lowest_priority();
				const int d = get_highest_priority() - l;
				return l + d / 2;
			}			
			inline static int get_normal_priority() 
			{ 
				static const int priority = do_get_normal_priority();
				return priority; 
			}
			inline static int do_get_above_normal_priority()
			{
				const int h = get_highest_priority();
				const int d = h - get_lowest_priority();
				return h - d / 4;
			}			
			inline static int get_above_normal_priority() 
			{ 
				static const int priority = do_get_above_normal_priority();
				return priority; 
			}
			inline static int get_highest_priority() { return sched_get_priority_max(get_default_schedule_policy()); }
			inline static int get_priority() { return get_priority(pthread_self()); }
			inline static int get_priority(pthread_t t_) 
			{ 
				sched_param sp;
				int policy;
				if(pthread_getschedparam(t_, &policy, &sp) != 0)
				{
					// error
					sp.sched_priority = 0;
				}
				return sp.sched_priority; 
			}
			inline static bool set_priority(int fPriority) { return set_priority(pthread_self(), fPriority); }
			static bool set_priority(pthread_t t_, int fPriority) 
			{ 
				sched_param sp;
				int policy;
				if(pthread_getschedparam(t_, &policy, &sp) != 0)
					return false;
				
				sp.sched_priority = fPriority;
				
				if(pthread_setschedparam(t_, policy, &sp) != 0)
					return false;
				
				return 0;				
			}

		};
	}
}

#endif
