/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
 
#ifndef WIN32_THREAD_H
#define WIN32_THREAD_H

#include <portablethreads/config.h>
#include <portablethreads/exception.h>
#include <memory>
#include <windows.h>

namespace PortableThreads 
{
	namespace OSSpecific
	{
		struct ThreadTraits
		{
			typedef struct { HANDLE handle_; DWORD id_; } thread_t;
			typedef DWORD os_id_t;
			typedef DWORD (WINAPI os_entry_function_t)(void*);
			typedef void (*normalized_entry_function_t)(void*);
			static DWORD LIBPORTABLETHREADS_API WINAPI wrap_os_entry_function(void*);
			struct normalized_entry_function_parameters_t
			{
				normalized_entry_function_t entry_;
				void* argument_;
			};
			inline static void create(thread_t& t, normalized_entry_function_t entry, void* arg)
			{
				DWORD temp;
				normalized_entry_function_parameters_t parameters = { entry, arg };
				const HANDLE h = CreateThread(0, 0, wrap_os_entry_function, &parameters, 0, &temp);
				if(h == NULL)
					throw PTResourceError("[PThread] Could not create thread");
				
				t.handle_ = h;
				t.id_ = temp;
			}
			inline static HANDLE create(os_entry_function_t* entry, void* arg, DWORD* fID = 0)
			{
				DWORD dummy;
				const HANDLE h = CreateThread(0, 0, entry, arg, 0, fID ? fID : &dummy);
				if(h == NULL)
					throw PTResourceError("[PThread] Could not create thread");
				return h;
			}
			inline static void create(thread_t& t, os_entry_function_t* entry, void* arg)
			{
				t.handle_ = create(entry, arg, &t.id_);
			}
			inline static void kill(thread_t& t)
			{
				TerminateThread(t.handle_, 0);
			}
			inline static void initialize(thread_t& t)
			{
				t.handle_ = NULL;
				t.id_ = 0;
			}
			inline static os_id_t self() { return GetCurrentThreadId(); }
			inline static void yield() { Sleep(0); }
			inline static void free_resource(thread_t& t)
			{
				CloseHandle(t.handle_);
			}
			inline static bool equal(os_id_t lhs, os_id_t rhs)
			{
				return lhs == rhs;
			}
			inline static os_id_t thread_id(const thread_t& t)
			{
				return t.id_;
			}
			inline static int get_lowest_priority() { return THREAD_PRIORITY_LOWEST; }
			inline static int get_below_normal_priority() { return THREAD_PRIORITY_LOWEST; }
			inline static int get_normal_priority() { return THREAD_PRIORITY_NORMAL; }
			inline static int get_above_normal_priority() { return THREAD_PRIORITY_ABOVE_NORMAL; }
			inline static int get_highest_priority() { return THREAD_PRIORITY_HIGHEST; }
			inline static int get_priority() { return GetThreadPriority(GetCurrentThread()); }
			inline static int get_priority(const thread_t& t_) { return GetThreadPriority(t_.handle_); }
			inline static bool set_priority(int fPriority) { return SetThreadPriority(GetCurrentThread(), fPriority) != 0; }
			inline static bool set_priority(const thread_t& t_, int fPriority) { return SetThreadPriority(t_.handle_, fPriority) != 0; }

		};
	}

	// No comparison operators for os_id_t -> DWORD builtin

}


#endif
