/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <portablethreads/thread.h>
#ifdef PT_WINDOWS
#	include <portablethreads/win32/thread.cpp>
#endif
#ifdef PT_UNIX
#	include <portablethreads/unix/thread.cpp>
#endif
#include <portablethreads/semaphore.h>
#include <portablethreads/utility.h>
#include <portablethreads/mutex.h>
#include <portablethreads/time.h>
#include <portablethreads/lockfree/stack.h>
#include <cstdio>
#include <cassert>

// Fix for windows.h include
#ifdef min
#	undef min
#endif 

using namespace std;

namespace PortableThreads 
{
	
	namespace
	{
		typedef OSSpecific::ThreadTraits OSThreadTraits;
		class ThreadControlBlock;
		typedef LockFree::PTStack<ThreadControlBlock*> ControlBlockList;

		class ThreadControlBlock
		{
			typedef PTGuard<PTMutex> Guard;
		public:
			typedef ControlBlockList::iterator iterator;
			ThreadControlBlock(iterator position)
				:	position_(position)
				,	joinCounter_(0)
				,	revision_(0)
				,	state_(PThread::DESTROYED)
			{
				OSThreadTraits::initialize(thread_);
			}
			inline const iterator& position() const { return position_; }
			inline int state() const { return state_; }
			inline void stop()
			{
				Guard guard(stateGuard_);
				if(state_ == PThread::RUNNING)
					stop_internal();
			}
			inline void resurrect()
			{
				assert(state_ == PThread::DESTROYED);
				state_ = PThread::STOPPED;
				assert(joinCounter_ == 0);
				++revision_;
			}
			inline int revision() const { return revision_; }
			void destroy()
			{
				Guard guard(stateGuard_);
				assert(state_ != PThread::DESTROYED);
				
				if(state_ == PThread::RUNNING)
					stop_internal();	
			
				state_ = PThread::DESTROYED;
				joinCounter_ = 0;
			}
			int join(int revision)
			{
				
				// NOTE: This is NOT synchronized!
				if(OSThreadTraits::equal(OSThreadTraits::thread_id(thread_), OSThreadTraits::self()))
					return PThread::DEADLOCK;


				// It could happen that some thread A is joining on 
				// this particular instance T and is delayed before
				// incrementing the counter in wait(). This is
				// not a problem, however, b/c T may either
				// be running (optimal case) or stopped.
				// I case T is running, the calling thread A
				// increments the counter and waits on the sem,
				// otherwise it will return without waiting. This
				// scheme will also work if A is delayed indefinitely. 
				// A will then eventually join on some other thread
				// T+n but it will still on on the SAME object.
				// If, however, the object containing T...T+n is destroyed
				// while A is still in join() but not in wait, the scheme will FAIL!
				// This isn't a problem b/c it would imply that A is calling a method on an
				// object (other than delete this) whose d'tor is called
				// before A's call returns -> Clearly not our problem!
				
				return wait(revision);
			}
			int run(OSThreadTraits::os_entry_function_t* entry, void* arg, int revision)
			{
				Guard guard(stateGuard_);
				
				if(revision != revision_)
					return PThread::ABANDONED;

				if(state_ == PThread::RUNNING)
					return PThread::ALREADY_RUNNING;
				
				assert(state_ == PThread::STOPPED);
				assert(joinCounter_ == 0);
				state_ = PThread::RUNNING;

				// NOTE: the next function may throw an exception!
				OSThreadTraits::create(thread_, entry, arg);
				return PThread::OK;
			}
		private:
			ThreadControlBlock();
			void stop_internal()
			{
				// Here the mutex must be locked!
				assert(stateGuard_.tryLock() == false);

				assert(state_ == PThread::RUNNING);
				state_ = PThread::STOPPED;

				releaseJoiners();
				OSThreadTraits::free_resource(thread_);
			}
			int wait(int revision)
			{
				stateGuard_.lock();
				if(state_ == PThread::STOPPED || state_ == PThread::DESTROYED)
				{
					stateGuard_.unlock();
					return PThread::NOT_RUNNING;
				}

				if(revision != revision_)
				{
					stateGuard_.unlock();
					return PThread::ABANDONED;
				}


				// Counter is used as a marker when to stop waiting
				// and as a counter to see how many threads are waiting
				// on this thread to die.
				// If counter < 0 then no more waiters are admitted b/c
				// the tread is terminating.
				// Otherwise inc the counter and block the calling 
				// thread on the semaphore.
				
				const bool inc = joinCounter_ >= 0;
				joinCounter_ += inc;
				stateGuard_.unlock();

				if(inc)
				{
					//printf("Waiting on join sem\n");
					sem_.down();
					if(revision != revision_)
						return PThread::ABANDONED;
					//printf("Waiting on join done\n");
				}
				//printf("join return\n");
				return PThread::OK;
			}
			void releaseJoiners()
			{
				const int current = joinCounter_;
				joinCounter_ = -1;
				assert(current >= 0);
				//printf("Releasing %d joiners\n", current);
				for(int i = 0; i < current; ++i)
					sem_.up();
				//printf("Releasing done\n");
			}
		private:
			OSThreadTraits::thread_t thread_;
			const iterator position_;
			volatile int joinCounter_;
			volatile int revision_;
			volatile int state_;
			PTSemaphore sem_;
			PTMutex stateGuard_;
		};

		/***********************************************************************/
		/* ThreadManager                                                       */
		/***********************************************************************/
		class ThreadManager
		{
		public:
			ThreadManager(unsigned reserve = 0);
			~ThreadManager();
			ThreadControlBlock* create();
			int destroy(PThread& thread);
			int poll(PThread::thread_id_type id) const;
			static ThreadManager& instance()
			{
				static ThreadManager instance_(4);
				return instance_;
			}
		private:
			ThreadControlBlock* getFreeControlBlock();
			void freeControlBlock(ThreadControlBlock* block);
		private:
			ControlBlockList live_, dead_;
		};

		ThreadManager* createInstanceBeforeMain = &ThreadManager::instance();

		ThreadManager::ThreadManager(unsigned reserve)
		{
			for(unsigned i = 0; i < reserve; ++i)
			{
				live_.push(0);
				dead_.push(new ThreadControlBlock(live_.begin()));
			}
		}
		ThreadManager::~ThreadManager()
		{
			//printf("ThreadManager dtor\n");
			// Wait for those threads that have deleted themselves and
			// are not yet out of the destroy function
			bool haslive = false;
			do
			{
				for(ControlBlockList::iterator it = live_.begin();
					it != live_.end(); ++it)
				{
					haslive = *it != 0;
					if(haslive)
					{
						pt_milli_sleep(1);
						break;
					}
				}
			}
			while(haslive);

			for(ControlBlockList::iterator it = dead_.begin();
				it != dead_.end(); ++it)
			{
				delete *it;
			}
			//printf("ThreadManager dtor END\n");
		}
		// called on PThread object creation to get a thread control block
		ThreadControlBlock* ThreadManager::create()
		{
			ThreadControlBlock* block = getFreeControlBlock();
			block->resurrect();
			//std::printf("Resurrecting thread %ul\n", (unsigned long)block);
			return block;
		}
		// called in PThread dtor to release thread control block
		int ThreadManager::destroy(PThread& thread)
		{
			assert(thread.threadId());
			ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(thread.threadId());
			
			block->destroy();
			freeControlBlock(block);
			return PThread::OK;
		}
		int ThreadManager::poll(PThread::thread_id_type id) const
		{
			if(!id)
				return PThread::DESTROYED;

			ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(id);
			return block->state();
		}

		ThreadControlBlock* ThreadManager::getFreeControlBlock()
		{
			ThreadControlBlock* block = 0;
			if(!dead_.pop(block))
			{
				block = new ThreadControlBlock(live_.push(0));
			}
			
			assert(block);
			assert(block->state() == PThread::DESTROYED);
			*block->position() = block;
			return block;
		}
		void ThreadManager::freeControlBlock(ThreadControlBlock* block)
		{
			assert(block->state() == PThread::DESTROYED);
			*block->position() = 0;
			dead_.push(block);
		}
	}

		


	/***********************************************************************/
	/* PThread                                                             */
	/***********************************************************************/

	const int PThread::OK = 0;
	const int PThread::ALREADY_RUNNING = -1;
	const int PThread::NOT_RUNNING = -2;
	const int PThread::DEADLOCK = -3;
	const int PThread::ABANDONED = -4;
	const int PThread::RUNNING = 1;
	const int PThread::STOPPED = 2;
	const int PThread::DESTROYED = 3;

	//std::auto_ptr<Private::ThreadManager> PThread::manager_(new Private::ThreadManager(4));

	void PThread::entry_common(void* arg)
	{
		PThread* thread = static_cast<PThread*>(arg);
		
		volatile bool objectGone = 0;
		thread->setObjectDestructionNotifier(&objectGone);
		try
		{
			thread->preThreadMain();
			thread->threadMain();			
		}
		catch(...)
		{
			thread->unexpectedException();
		}
		// If a thread deletes itself in its
		// dtor, which will be invoked before this
		// function returns it will set notify 
		// this function via objectGone so
		// "stop()" isn't called on a dead
		// thread (not so bad) but also on
		// a totally different thread object 
		// resurrected thread_id.
		if(!objectGone)
		{
			thread->setObjectDestructionNotifier(0);
			reinterpret_cast<ThreadControlBlock*>(thread->threadId())->stop();
		}
	}
	int PThread::poll(thread_id_type id)
	{
		return ThreadManager::instance().poll(id);
	}

	PThread::~PThread()
	{ 
		// Currently it is not possible to join a 
		// thread by invoking it's dtor b/c I have not 
		// found a way to determine whether the thread
		// is deleting itself or an external caller
		// caused the d'tor to be called. While in
		// the first case it is ok to assume the thread
		// is about to die anyway, this is most likely
		// not the case when invoking from the outside ->
		// free a running object's resources is a bad 
		// idea, b/c the thread might leave threadMain()
		// and then try to access it's members in static entry
		// via the base class pointer. However, members may
		// already been gone b/c the main thread could have
		// succeeded with PThread's dtor. 
		// If we disable object deletion notification 
		// whenever invoking a PThread's dtor there 
		// is a race condition with the "object gone" 
		// variable in the stack of static entry function.
		// The thread in static entry might read the value
		// determining access is ok while the main thread
		// destroys the object in between -> ZAP
		// ALSO: When trying to join via dtor and the thread
		// has not already entered threadMain (which is overwritten)
		// in the derived class, the dtor invokation will
		// cause the d'tor of the derived class to be called
		// emptying the pure virtual function slot ->
		// when a thread then calls threadMain to start 
		// running -> ZAP. Other than that the derived
		// classes members are gone, of course whenever
		// the dtor is called on a PThread object.

		// Hence: for now it you shouldn't invoke a threads
		// dtor before the thread is really stopped
		if(notifyOnDestruction_)
			*notifyOnDestruction_ = true;
		ThreadManager::instance().destroy(*this);
	}

	PThread::PThread()
		:	notifyOnDestruction_(0)
	{
		ThreadControlBlock* block = ThreadManager::instance().create();
		id_ = block;
		revision_ = block->revision();
	}

	int PThread::run()
	{
		assert(id_);
		return reinterpret_cast<ThreadControlBlock*>(id_)->run(&PThread::entry, this, revision_);
	}
	int PThread::join()
	{
		assert(id_);
		return reinterpret_cast<ThreadControlBlock*>(id_)->join(revision_);
	}

	void PThread::threadMain()
	{
		std::printf("[PThread] Pure virtual method called!\n");
	}
	void PThread::unexpectedException() throw()
	{
		std::printf("[PThread] Caught unhandled exception.\n");
	}
	void PThread::threadId(thread_id_type id)
	{
		id_ = id;
	}
	void PThread::setObjectDestructionNotifier(volatile bool* notifiy)
	{
		notifyOnDestruction_ = notifiy;
	}
	void PThread::give() const
	{
		OSThreadTraits::yield();
	}
	PThread::thread_id_type PThread::threadId() const
	{
		return id_;
	}
	void PThread::preThreadMain()
	{}

	/***********************************************************************/
	/* PTSuspendableThread                                                 */
	/***********************************************************************/

	
	PTSuspendableThread::PTSuspendableThread()
		:	blocker_(true)
		,	lockFlag_(1)
	{
	}
	bool PTSuspendableThread::resume()
	{
		const bool ret = lockFlag_.cas(0, 1);
		if(ret)
			blocker_.unlock();
		return ret;
	}
	void PTSuspendableThread::suspend()
	{
		if(lockFlag_.cas(1, 0))
			blocker_.lock();
	}
	bool PTSuspendableThread::suspended() const
	{
		return lockFlag_.get() == 1;
	}
	void PTSuspendableThread::preThreadMain()
	{
		blocker_.lock();
	}
	
	/***********************************************************************/
	/* PThreadRunner                                                       */
	/***********************************************************************/
	
	void  PThreadRunner::start(function_adaptor_t fThreadFunc, function_adaptor_t fNotifyWhenDoneFunc)
	{
		if(!fThreadFunc.get())
			return; // ### OUT ###
			
		if(!registeredWaitHandler_) // static initialization order
			registeredWaitHandler_ = registerWaitHandler();

		PThreadRunner* runner = new PThreadRunner(fThreadFunc, fNotifyWhenDoneFunc);
		try
		{
			ThreadTraits::create(
				PThreadRunner::entry, 
				runner);
			
			++threadsToWaitFor_;
		}
		catch(...)
		{
			delete runner;
			throw;
		}
	}
	
	int PThreadRunner::getPriority()
	{
		const int prio = ThreadTraits::get_priority();
		if(prio == ThreadTraits::get_lowest_priority())
			return PRIO_LOWEST;
		else if(prio == ThreadTraits::get_below_normal_priority())
			return PRIO_BELOW_NORMAL;
		else if(prio == ThreadTraits::get_above_normal_priority())
			return PRIO_ABOVE_NORMAL;
		else if(prio == ThreadTraits::get_highest_priority())
			return PRIO_HIGHEST;

		return PRIO_NORMAL;
	}
	bool PThreadRunner::setPriority(int fPrio)
	{
		switch(fPrio)
		{
		case PRIO_LOWEST :
			return ThreadTraits::set_priority(ThreadTraits::get_lowest_priority());
		case PRIO_BELOW_NORMAL :
			return ThreadTraits::set_priority(ThreadTraits::get_below_normal_priority());
		case PRIO_NORMAL :
			return ThreadTraits::set_priority(ThreadTraits::get_normal_priority());
		case PRIO_ABOVE_NORMAL :
			return ThreadTraits::set_priority(ThreadTraits::get_above_normal_priority());
		case PRIO_HIGHEST :
			return ThreadTraits::set_priority(ThreadTraits::get_highest_priority());
		}

		return false;
	}

	PThreadRunner::PThreadRunner(function_adaptor_t fThreadFunc, function_adaptor_t fThreadDoneFunc)
		:	functionToRun_(fThreadFunc)
		,	functionToCallWhenDone_(fThreadDoneFunc)
	{}
	void PThreadRunner::run()
	{
		const_cast<function_adaptor_t&>(functionToRun_)->invoke();
		if(const_cast<function_adaptor_t&>(functionToCallWhenDone_).get())
			const_cast<function_adaptor_t&>(functionToCallWhenDone_)->invoke();	
		
		delete this;
	}
	
	void PThreadRunner::entry_common(void* fRunner)
	{
		LockFree::Private::pt_mfence(); // ensure data visibility
		try
		{
			PThreadRunner* runner = static_cast<PThreadRunner*>(fRunner);
			runner->run();
		}
		catch(...)
		{
			fprintf(stderr, "[PThreadRunner] Thread terminates gracefully after catching uncaught exception from user code.");
			fflush(stderr);
		}
		--threadsToWaitFor_;
	}
	
	void PThreadRunner::joinRunningThreads() { while(threadsToWaitFor_.get() > 0) pt_yield(); }
	bool PThreadRunner::registerWaitHandler() { std::atexit(joinRunningThreads); return true; }

	
	LockFree::PTAtomicNumber PThreadRunner::threadsToWaitFor_;
	bool PThreadRunner::registeredWaitHandler_ = PThreadRunner::registerWaitHandler();
}

