/*	$Id: main.cpp 8142 2007-10-24 14:19:52Z rtichy $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <unistd.h>
#include <iostream>
#include <memory>
#include <stdexcept>
#include <algorithm>
#include <iterator>
#include <platypus/platypus.h>
#include <platypus/utility/options_faker.h>

#ifdef _MSC_VER
	#include <crtdbg.h>
#endif

using namespace std;
using namespace Platypus;

void print_initialization();
void print_registered_types(const PlatypusTypesFactory& factory);

int main(int argc, char** argv)
{
#ifdef _MSC_VER
  if(1)
    {
      _CrtSetDbgFlag(_CrtSetDbgFlag(_CRTDBG_REPORT_FLAG) |
		     _CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF |
		     _CRTDBG_CHECK_ALWAYS_DF);
      _CrtSetReportMode( _CRT_WARN, _CRTDBG_MODE_FILE );
      _CrtSetReportFile( _CRT_WARN, _CRTDBG_FILE_STDERR );
      _CrtSetReportMode( _CRT_ERROR, _CRTDBG_MODE_FILE );
      _CrtSetReportFile( _CRT_ERROR, _CRTDBG_FILE_STDERR );
      _CrtSetReportMode( _CRT_ASSERT, _CRTDBG_MODE_FILE );
      _CrtSetReportFile( _CRT_ASSERT, _CRTDBG_FILE_STDERR );
    }
#endif
  
  PlatypusTypesFactory& factory = PlatypusTypesFactory::instance();
  PlatypusBuilder builder;
  try
    {
      //print_initialization();
      factory.loadModules(argc, argv);
      //print_registered_types(factory);
      
      builder.build(argc, argv);
      
      auto_ptr<PlatypusAlgorithm> platypus(builder.sellProduct());
      if(!platypus.get()) // user wants help on options
	{
	  cout	<< builder.usage() << "\n" 
		<< OptionsFaker::instance().options() << "\n" 
		<< factory.usage() << endl;
	  return 1;
	}
      
      // if there is a signal catch it, 
      // disable further signals and 
      // call shutdown on platypus object
      Platypus::notifyOnSignal(*platypus);
      
      platypus->run();
      cout << *platypus;
      cout.flush();
    }
  catch(const CommandlineError& e) // wrong use
    {
      cerr << e.what() << "\n\n" << builder.usage() << "\n";
      cerr << OptionsFaker::instance().options() << "\n";
      cerr << factory.usage();
      return 1;
    }
  catch(const PlatypusException& e)
    {
      cerr << e.what() << endl;
      return 1;
    }
  catch(const exception& e)
    {
      cerr << e.what() << endl;
      return 1;
    }
  catch(...)
    {
      cerr << "Unknown error, exiting" << endl;
      return 1;
    }
  return 0;
}

void print_initialization() 
{
  cout << "Initializing module load..." << endl;
}

void print_registered_types(const PlatypusTypesFactory& factory)
{
  {
    DistributionFactory::key_chain_type keys = factory.distributionFactory().keys();
    cout << "Registered distribution keys: ";
    copy(keys.begin(), keys.end(), ostream_iterator<string>(cout, " "));
    cout << endl;
  }
  {
    ExpanderFactory::key_chain_type keys = factory.expanderFactory().keys();
    cout << "Registered expander keys: ";
    copy(keys.begin(), keys.end(), ostream_iterator<string>(cout, " "));
    cout << endl;
  }
  {
    ProgramFactory::key_chain_type keys = factory.programFactory().keys();
    cout << "Registered program keys: ";
    copy(keys.begin(), keys.end(), ostream_iterator<string>(cout, " "));
    cout << endl;
  }
}








