/*	$Id: fork_serializer.h 1771 2005-05-16 12:00:38Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef FORK_SERIALIZER_H
#define FORK_SERIALIZER_H

#include <cassert>
#include <platypus/types_fwd.h>
#include <platypus/types/choice.h>
#include <distribution/serializer_base.h>
#include <distribution/fork/ipc.h>

namespace Platypus
{
  class ForkSerializer : SerializerBase
    {
    public:
      
      ForkSerializer(const ProgramInterface& program) : program_(&program) {}
      ~ForkSerializer() {}
      // returns the number of bytes required to send complete assignment
      size_t bytesRequired() const
      {
	// [size, <Atom id, truthvalue>, <...>, ...], all of type AtomId
	return (1 + program_->numberOfAtoms() * 2) * sizeof(AtomId);
      }
      template<class T> void deserialize(DelegatableChoice& dc, const T& buffer) const;
      template<class T> void serialize(T& buffer, const DelegatableChoice& dc) const;
      
      template<class T> void deserialize(PartialAssignment& pa, const T& buffer) const;
      template<class T> void serialize(T& buffer, const PartialAssignment& pa) const;
    private:
      ForkSerializer();
    private:
      const ProgramInterface* program_;
    };
  
  template<class T>
    void ForkSerializer::deserialize(DelegatableChoice& dc, const T& buffer) const
    {
      assert(buffer.size() >= sizeof(AtomId));
      
      const AtomId* ids = reinterpret_cast<const AtomId*>(buffer.data());
      const AtomId size = ids[0];
      //std::printf("D dc size: %d\n", size);
      // [size, <Atom id, truthvalue>, <...>, ...], all of type AtomId
      assert(buffer.size() >= sizeof(AtomId)*(2*size+1));
      
      dc.clear();
      size_t index = 1;
      for(size_t i = 0; i < size; ++i, index += 2)
	{
	  dc.push_back(Choice(ids[index], ids[index+1]));
	}
  }
  
  template<class T>
    void ForkSerializer::serialize(T& buffer, const DelegatableChoice& dc) const
    {
      // [size, <Atom id, truthvalue>, <...>, ...], all of type AtomId
      const size_t size = dc.size();
      buffer.resize((1+dc.size()*2)*sizeof(AtomId));
      AtomId* ids = reinterpret_cast<AtomId*>(buffer.data());
      ids[0] = static_cast<AtomId>(size);
      
      size_t index = 1;
      for(size_t i = 0; i < size; ++i, index += 2)
	{
	  ids[index] = dc[i].id();
	  ids[index+1] = dc[i].isPositive();
	}
    }
  
  template<class T>
    void ForkSerializer::deserialize(PartialAssignment& pa, const T& buffer) const
    {
      assert(buffer.size() >= sizeof(AtomId));		
      const AtomId* ids = reinterpret_cast<const AtomId*>(buffer.data());
      const size_t size = static_cast<size_t>(ids[0]);
      assert(buffer.size() >= sizeof(AtomId)*(1+size));
      
      size_t index = 1; pa.clear();
      for(size_t i = 0; i < size; ++i, ++index)
	{
	  pa.setTrue(Atom(ids[index], *program_));
	}
    }

  template<class T>
    void ForkSerializer::serialize(T& buffer, const PartialAssignment& pa) const
    {
      const size_t positive = pa.positiveAtoms().size();
      
      buffer.resize(sizeof(AtomId)*(positive+1));
      AtomId* ids = reinterpret_cast<AtomId*>(buffer.data());
      ids[0] = positive;
      
      size_t index = 1;
      const PartialAssignment::CollectionType::const_iterator END = pa.positiveAtoms().end();
      for(PartialAssignment::CollectionType::const_iterator it = pa.positiveAtoms().begin();
	  it != END; ++it, ++index)
	{
	  ids[index] = it->id();
	}
    }
}



#endif
