/*	$Id: fork_serializer.cpp 1782 2005-05-17 14:52:01Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef MPI_SERIALIZER_H
#define MPI_SERIALIZER_H

#include <iostream>
#include <mpi.h>
#include <cstdio>
#include <cassert>
#include <vector>
#include <platypus/types_fwd.h>
#include <platypus/types/choice.h>
#include <platypus/types/atom.h>
#include <platypus/types/partial_assignment.h>
#include <distribution/serializer_base.h>
#include <distribution/mpi/mpi_platypus_types.h>
#include <distribution/mpi/mpi_message_buffer.h>

namespace Platypus
{

  /**
   * Handles the translation of choices and partial assignments into
   * into mpi specific message buffers and visa versa.
   *
   */  
  class MPISerializer : public SerializerBase
  {
  public:
    MPISerializer(const ProgramInterface& program) : program_(&program) {}
    ~MPISerializer() {}
    // returns the number of bytes required to send complete assignment
    size_t bytesRequired() const
    {
      return (1 + (program_->numberOfAtoms() * 2)); 
    }
    
    // remove a message from the message buffer and place it in a choice object
    template<class T> void deserialize(DelegatableChoice& dc, const T& buffer) const;
    
    // translate a choice into a message buffer for transfer from
    // worker to worker
    template<class T> void serialize(T& buffer, const DelegatableChoice& dc) const;
    
    // following two methods are same as above but for moving
    // answer sets around which are stored in PartialAssigment objects
    template<class T> void deserialize(PartialAssignment& pa, const T& buffer) const;
    
    template<class T> void serialize(T& buffer, const PartialAssignment& pa) const;
    
  private:
    MPISerializer() {};
    
  private:
    const ProgramInterface* program_;
  };
  
  
  // remove a message from the message buffer and place it in a choice object
  template<class T> 
    void MPISerializer::deserialize(DelegatableChoice& dc, const T& buffer) const
    {
      const unsigned size = buffer.size();
      dc.clear();
      
      const unsigned long * data =  reinterpret_cast<const unsigned long*>(buffer.data());
      
      for(size_t i = 0; i < size; i += 2)
	{
	  dc.push_back(Choice(data[i], data[i+1]));
	}
      
    }
  
  // translate a choice into a message buffer for transfer from
  // worker to worker
  template<class T> 
    void MPISerializer::serialize(T& buffer, const DelegatableChoice& dc) const
    {
      //find the size of the dc... it is simply a typedefed std container (vector of choices)
      //this gives the number of choices, each which contains an id and a boolean value. To send 
      //this via a message you need to add both values to the message.
      const size_t size = dc.size();
      if(buffer.size() != (size << 1))
	buffer.resize(size << 1);
      
      unsigned long * data =  reinterpret_cast<AtomId*>(buffer.data());
      
      for(size_t i = 0; i < size; i++)
	{
	  unsigned index = i << 1;//* 2;
	  data[index] = dc[i].id();	
	  data[index+1] = dc[i].isPositive();
	}
      
    }
  
  // following two methods are same as above but for moving
  // answer sets around which are stored in PartialAssigment objects
  template<class T> 
    void MPISerializer::deserialize(PartialAssignment& pa, const T& buffer) const
    {
      
      const unsigned size = buffer.size();
      const unsigned long * data =  reinterpret_cast<const unsigned long*>(buffer.data());
      pa.clear();
      for(size_t i = 0; i < size; ++i)
	{
	  pa.setTrue(Atom(data[i], *program_));
	}
      
      
    }
  
  template<class T> 
    void MPISerializer::serialize(T& buffer, const PartialAssignment& pa) const
    {
      
      const size_t size = pa.positiveAtoms().size();

      if(buffer.size() != size)
        buffer.resize(size); 

      unsigned long * data =  reinterpret_cast<unsigned long*>(buffer.data());
      size_t index = 0;
      
      const PartialAssignment::CollectionType::const_iterator END = pa.positiveAtoms().end();
      for(PartialAssignment::CollectionType::const_iterator it = pa.positiveAtoms().begin();
	  it != END; ++it, ++index)
	{
	  data[index] = it->id();
	}
      
    }
}
#endif
