//
//	(C) Copyright Benjamin Kaufmann, 2004-2005.
//	Permission to copy, use, modify, sell and distribute this software is
//	granted provided this copyright notice appears in all copies.
//	This software is provided "as is" without express or implied warranty,
//	and with no claim as to its suitability for any purpose.
//
#ifndef EVENT_CHANNEL_H_INCLUDED
#define EVENT_CHANNEL_H_INCLUDED
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif
#include <vector>
#include <deque>
namespace event { namespace detail {

// classes in the detail namespace are for internal use only

// base interface for all Channels.
// Required by the ChannelManager
struct ChannelBase
{
	virtual ~ChannelBase() = 0;
};

}	// end of namespace event::detail


/*!
 * \brief The class CallTraits defines the mapping between an event type
 * and the member function of an event handler that should handle events of that type.
 *
 * The mapping is used by an \ref Channel "event channel" whenever it
 * dispatches events to their handlers.
 *
 * The primary template maps events of type E to member functions H::handle(const E&).
 *
 * Specialize this class in order to define other mappings for concrete event
 * and handler types.
 *
 * \par Example:
 * \code
 * struct AHandler
 * {
 *   void onEvent(const AnEvent&) const;
 * };
 * namespace event {
 * // dispatch events of type AnEvent to AHandler's onEvent function.
 * template <>
 * struct CallTraits<AHandler, AnEvent> {
 *   static void call(H& r, const E& e) {
 *     r.onEvent(e);
 *   }
 * };
 * }
 * \endcode
 */
template <class T, class E>
struct CallTraits
{
	static void handle(T& o, const E& e)
	{
		o.handle(e);
	}
};

/*!
 * \brief The class EventDispatchingState defines different event processing
 * states that determine the way an \ref Channel "event channel" dispatches events.
 */
struct EventDispatchingState
{
	enum State
	{
		LIFO,	/*!< Process events in LIFO order, ie. if an event e2 arrives
				 * while a channel dispatches an event e1, the dispatch of e1
				 * is interrupted and e2 is dispatched first.
				 */
		FIFO,	/*!< Process events in FIFO order. A channel in this state always
				 * dispatches an event to all handlers before it starts dispatching
				 * new events. Events that arrive in the meantime are queued.
				 */
		HOLD,	/*!< In the Hold-State a channel only queues events but does not dispatch them.
				 */
		DISCARD	/*!< In the Discard-State a channel discards all arriving events. */
	};
};

typedef EventDispatchingState::State EventDispatchingState_t;

/*!
 * \brief Objects of the class Channel are used to transfer events
 * of type EventType from event sources to event sinks (aka. handlers).
 *
 * Channels are strongly typed, i.e. a channel can only transfer events
 * of one type.
 *
 * \par Requirements for EventType:
 * \n
 * EventType must be assignable if Channel is to be used in
 * \ref EventDispatchingState::FIFO "FIFO" or \ref EventDispatchingState::HOLD "HOLD" mode
 *
 * \note Channel trades maximum dispatch performance in
 * \ref EventDispatchingState::LIFO "LIFO" mode against lower performance for other
 * operations. For example disconnecting a handler while an event is active
 * is a costly operation because it has to walk through the handler list
 * twice (once to find and mark the to be disconnected handler and once to actually
 * remove the handler when the event is no longer active).
 */
template <class EventType>
class Channel : public detail::ChannelBase
{
public:

	/*!
	 * \brief Creates a new Channel.
	 *
	 * \param s The EventDispatchingState used by the new channel.
	 */
	explicit Channel(EventDispatchingState_t s = EventDispatchingState::LIFO)
		: state_(s)
	{
		flags_.eventActive_ = false;
		flags_.inQueueMode_ = false;
		flags_.deferredDisconnect_ = false;
	}

	/*!
	 * \brief Destroys the channel and disconnects all connected Handlers.
	 */
	~Channel()
	{}


	/*!
	 * \brief Registers obj as Handler for succeeding events of type EventType.
	 *
	 * \complexity Constant time.
	 *
	 * \note The word succeeding in this case means, that handlers that connect
	 * while an event is active are *not* notified about the currently active event
	 * but only about new events.
	 *
	 * \par Rationale:
	 * Handlers can connect at any point in time, including while those Handlers
	 * are being invoked. If new Handlers were notified about active events
	 * new connections could result in infinite loops.
     *
	 * \note The Channel stores only a pointer to the handler.
	 * To avoid dangling references make sure to disconnect the handler
	 * before its lifetime ends.
	 *
	 * \par Requirements for T:
	 * \n T has an appropriate handler-function (see CallTraits)
	 */
	template <class T>
	void connect(T& obj)
	{
		SubscriberList& l = flags_.eventActive_ ? deferredConnects_ : subscriberList_;
		l.push_back(SubscriberInfo(&obj, &Thunk<T>::thunk));
	}

	/*!
	 * \brief Disconnects obj from this channel.
	 *  Events (including a currently active event) will no longer be dispatched to obj.
	 *
	 * \complexity Linear in the number of Handlers connected to the Channel.
	 * \note If an event is active disconnect has to walk through the list
	 * of connected Handlers twice.
	 */
	template <class T>
	void disconnect(T& obj)
	{
		for (typename SubscriberList::iterator it = subscriberList_.begin(); it != subscriberList_.end();)
		{
			if (it->handler_ == static_cast<void*>(&obj))
			{
				if (!flags_.eventActive_)
				{
					it = subscriberList_.erase(it);
				}
				else
				{	// an event is currently active.
					// To keep the iterators in the dispatch-loop valid we defer
					// the deletion and change obj's SubscriberInfo so that
					// it no longer calls back obj's handle method.
					// By setting handler_ to 0 we mark this SubscriberInfo
					// as disconnected. After the active event is fully
					// dispatched we'll delete all marked entries.
					it->handler_ = 0;
					it->thunk_ = &Channel::noop;
					flags_.deferredDisconnect_ = true;
					++it;
				}
			}
			else
				++it;
		}
	}

	/*!
	 * \brief returns the number of handlers connected to this channel.
	 *
	 * \complexity constant if no event is active,
	 *	linear in the number of connected handlers otherwise.
	 * \par Rationale:
	 * Handlers can disconnect at any point in time,
	 * including while those Handlers are being invoked.
	 * It is therefore possible that the implementation must search through a
	 * list of disconnected Handlers to determine how many Handlers are still connected.
	 *
	 */
	unsigned numHandlers() const
	{
		if (!flags_.eventActive_)
			return (unsigned int)subscriberList_.size();
		else
		{
			unsigned count = 0;
			for (typename SubscriberList::const_iterator it = subscriberList_.begin(); it != subscriberList_.end(); ++it)
				if (it->handler_)
					++count;
			return count;
		}
	}

	/*!
	 * \brief returns true if no Handlers are connected to this channel, and false otherwise.
	 *
	 * \complexity constant if no event is active and linear in the number of connected handlers otherwise.
	 * \see numHandlers
	 */
	bool hasHandlers() const
	{
		if (!flags_.eventActive_)
			return !subscriberList_.empty();
		return numHandlers() != 0;
	}

	/*!
	 * \brief dispatches the Event e in a way defined by the channel's current
	 * EventDispatchingState.
	 *
	 * \complexity linear in the number of connected handlers.
	 *
	 * \note Handlers are notified in the order they connected.
	 *
	 * \note If an exception is thrown by a Handler the dispatch process
	 * stops and succeeding Handlers are not notified.
	 */
	void dispatch(const EventType& e) const
	{
		if (preEvent(e))
		{
			typename SubscriberList::const_iterator end = subscriberList_.end();
			for (typename SubscriberList::const_iterator it = subscriberList_.begin(); it != end; ++it)
				it->thunk_(it->handler_, e);
			postEvent(e);
		}
	}

	/*!
	 * \brief returns the current \ref EventDispatchingState "EventDispatchingState"
	 * of this channel.
	 **/
	EventDispatchingState_t getEventDispatchingState() const
	{
		return state_;
	}

	/*!
	 * \brief sets a new \ref EventDispatchingState "EventDispatchingState".
	 *
	 * \note If the state changes from \ref EventDispatchingState::HOLD "HOLD"
	 * or \ref EventDispatchingState::FIFO "FIFO" to \ref EventDispatchingState::LIFO "LIFO"
	 * all currently queued events are dispatched.
	 * If the new state is \ref EventDispatchingState::DISCARD "DISCARD" currently
	 * queued events are also discarded.
	 *
	 */
	void setEventDispatchingState(EventDispatchingState_t s)
	{
		if (s == EventDispatchingState::LIFO
			&& (state_ == EventDispatchingState::HOLD || state_ == EventDispatchingState::FIFO))
		{
			state_ = s;
			dispatchQueue();
		}
		else if (s == EventDispatchingState::DISCARD)
		{
			state_ = s;
			eventQueue_.clear();
		}
		else
			state_ = s;
	}

	/*!
	 * \brief returns true, if the channel currently dispatches an event, and false otherwise.
	 */
	bool isEventActive() const
	{
		return flags_.eventActive_;
	}
private:
	Channel(const Channel&);				// channels don't support copying
	Channel& operator=(const Channel&);		//

	// SubscriberInfo holds the information needed to callback
	// an event handler.
	// The information is stored in a typeless manner so that arbitrary Handler types
	// can be stored in one handler list.
	// A generic thunk (aka trampoline) is used to restore the type information
	// whenever an event is dispatched.
	struct SubscriberInfo
	{
		void* handler_;
		void (*thunk_)(void* r, const EventType&);
		SubscriberInfo(void* r, void (*thunk)(void*, const EventType&))
			: handler_(r)
			, thunk_(thunk)
		{}
	};

	// encapsulates information about the type H
	template <class H>
	struct Thunk
	{
		// restores the type of h and then dispatches e to the
		// appropriate member funtion.
		static void thunk(void* h, const EventType& e)
		{
			event::CallTraits<H, EventType>::handle(*static_cast<H*>(h), e);
		}
	};

	// noop is used as callback function for handlers that
	// are disconnected while an event is active.
	static void noop(void*, const EventType&) {}

	// Using a list instead of a vector would allow for a faster implementation
	// for deferred disconnects (by saving iterators to the to be deleted elements).
	// On the other hand the dispatch-loop profits from the fact that
	// vector's iterators are generally leaner and faster then list's.
	typedef std::vector<SubscriberInfo> SubscriberList;

	typedef std::deque<EventType> EventQueue;

	// decides if e can currently be dispatched.
	bool preEvent(const EventType& e) const
	{
		if (state_ == EventDispatchingState::LIFO
			|| (state_ == EventDispatchingState::FIFO && !flags_.eventActive_))
		{
			flags_.eventActive_ = true;
			return true;
		}

		if (subscriberList_.empty() || state_ == EventDispatchingState::DISCARD)
			return false;

		// state is HOLD or FIFO and an event is active.
		eventQueue_.push_back(e);
		return false;
	}

	// executes deferred actions
	void postEvent(const EventType&) const
	{
		flags_.eventActive_ = false;
		processDeferredCommands();

		if (state_ == EventDispatchingState::FIFO && !flags_.inQueueMode_)
		{
			dispatchQueue();
		}
	}

	// executes deferred connects/disconnects
	void processDeferredCommands() const
	{
		if (flags_.deferredDisconnect_)
		{
			for (typename SubscriberList::iterator it = subscriberList_.begin(); it != subscriberList_.end();)
			{
				if (it->handler_ == 0)
					it = subscriberList_.erase(it);
				else
					++it;
			}
			flags_.deferredDisconnect_ = false;
		}
		if (deferredConnects_.size())
		{
			for (typename SubscriberList::iterator it = deferredConnects_.begin(); it != deferredConnects_.end(); ++it)
			{
				subscriberList_.push_back(*it);
			}
			deferredConnects_.clear();
		}
	}

	// dispatches all events stored in the event queue.
	void dispatchQueue() const
	{
		flags_.inQueueMode_ = true;
		while (!eventQueue_.empty() &&
			(state_ != EventDispatchingState::HOLD))
		{
			// copy the event. This is necessary because eventQueue_.front()
			// may become invalid (e.g. one handler sets the state to DISCARD).
			EventType e = eventQueue_.front();
			eventQueue_.pop_front();
			dispatch(e);
		}
		flags_.inQueueMode_ = false;
	}

	mutable SubscriberList	subscriberList_;
	mutable SubscriberList	deferredConnects_;
	mutable EventQueue		eventQueue_;
	EventDispatchingState_t	state_;
	struct
	{
        // indicates whether an event is currently active, i.e.
		// if there is an dispatch loop.
		mutable bool	eventActive_:1;

		// True if we are dispatching events out of our eventQueue.
		mutable bool	inQueueMode_:1;

		// True if some Handlers have disconnected, while an event was active.
		// In that case  the disconnect must be deferred in order to keep
		// iterators in the dispatch loop valid.
		mutable bool	deferredDisconnect_:1;
	} flags_;
};

}	// end of namespace event


#endif
