//
//	(C) Copyright Benjamin Kaufmann, 2004-2005. 
//	Permission to copy, use, modify, sell and distribute this software is 
//	granted provided this copyright notice appears in all copies. 
//	This software is provided "as is" without express or implied warranty, 
//	and with no claim as to its suitability for any purpose.
//
#ifndef EVENT_CHANNEL_MANAGER_H_INCLUDED
#define EVENT_CHANNEL_MANAGER_H_INCLUDED
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif
#include <event/channel.h>
#include <event/eventlist.h>
#include <vector>
#include <cassert>
/*!
 * \mainpage An implementation of the Typed message pattern<a href="#typed_message">[1]</a><a href="#typed_message2">[2]</a>
 *
 * \author <a href="mailto:kaufmann@cs.uni-potsdam.de">Benjamin Kaufmann</a>
 * \par Internet:
 * http://fara.cs.uni-potsdam.de/~kaufmann/
 * \date 2004-2005
 *
 * \section intro Introduction
 * This little framework is essentually an implementation of the Typed message
 * pattern <a href="#typed_message">[1]</a><a href="#typed_message2">[2]</a>.
 *
 * Typed message (and therefore this implementation) is useful when all of
 * the following are true (quoting from the applicability section):
 * - Certain classes of objects may be interested in receiving information from other objects.
 * - The information is of arbitrary structure and complexity and may vary as the software evolves.
 * - The information transfer should be statically type-safe.
 *
 * In his book <a href="#typed_message">"Pattern Hatching"</a>, John Vlissides
 * discusses two different ways for establishing the connection between events
 * and there sources. Event handlers either register with the event types
 * or with the event sources.
 * Both implementations have there problems. In the first variant it is difficult
 * to have different set of handlers for the same event type (because the event
 * type can't distinguish the handlers). The second
 * variant is wasteful if there are a great many event sources all producing
 * events of the same type and handlers are interested in the events of all sources.
 *
 * This implementation tackles both problems by introducing a mediator
 * between event sources and event handlers.
 * This mediator is called \ref event::Channel "event channel" and has two main
 * responsibilities:
 * - It provides an registration interface for event handlers.
 * - It provides an dispatch interface for event sources.
 *
 * Having this mediator in place the problems mentioned above are easily solved:
 * - If you need different sets of handlers create a channel for each set.
 * - If you have a great many event sources and you don't want to register
 * with each and every source, let your sources use the same channel and
 * register your handlers with that channel.
 *
 * All in all in this implementation one needs to distinguish five different participants:
 * \par 1. Event types:
 * \n
 * Event types are arbitrary value types that encapsulate information to
 * be transferred from event sources to event handlers.
 *
 * \par 2. Event channels:
 *	\n
 *	Event channels are objects of the class \ref event::Channel "Channel".
 *	An Event channel acts as a mediator between event sources and event handlers
 *	therefore keeping them from referring to each other explicitly.
 *	Event channels maintain references to event handlers and provide
 *	a dispatch method over which event sources can send information to
 *  interested objects.
 *
 * \par 3. Event sources:
 * \n
 * Event sources create events and send them to interested handlers by
 * calling event::Channel::dispatch.
 *
 * \par 4. Event handler:
 * \n
 *  Event handlers are arbitrary types that provide member functions
 *	of the form
 * \code
 *	void handle(const E&)
 * \endcode
 *	for each concrete event type E the handler is interested in (refer to the documentation of the class
 *	\ref event::CallTraits "CallTraits" for information on how to specify the
 *	names of the concrete handler methods).
 *	\n
 *	Event handlers connect to an \ref event::Channel "event channel" in order
 *	to receive events.
 *
 *
 * \par 5. Channel Manager:
 * \n
 * \ref event::ChannelMaanager "Channel Manager" manage and provide access
 * to \ref event::Channel "event channels" for several different event types.
 *
 * \par Minimal Example:
 * \code
 * struct AnEvent {};
 * struct AnotherEvent {};
 * struct AHandler {
 *   void handle(const AnEvent&) {
 *     cout << "received an event of type AnEvent" << endl;
 *   }
 *   void handle(const AnotherEvent&) {
 *     cout << "received an event of type AnotherEvent" << endl;
 *   }
 * };
 * ...
 * int main() {
 *   event::ChannelManager m;
 *   AHandler h;
 *   // connect h to the channels for AnEvent and AnotherEvent
 *   event::connectTo<event::EventList<AnEvent, AnotherEvent> >(m, h);
 *   // dispatch AnEvent to all interested handlers
 *   m.dispatch(AnEvent());
 * }
 * \endcode
 * \n
 * \par References:
 * \n
 * <a name="typed_message">[1]</a>: John Vlissides, Pattern Hatching, Addison-Wesley, 1998 \n
 * <a name="typed_message2">[2]</a>: <a href="http://www.research.ibm.com/designpatterns/pubs/typed-msg.html">John Vlissides, Multicast - Observer = Typed Message</a>
 */

/**
 * \namespace event
 * \brief Contains types and functions implementing a simple type safe event system.
 *
 * This event-Module contains types and functions implementing a simple type safe
 * event system. Events are arbitrary value types created by event
 * sources representing typed information that should be transfered between
 * different objects.
 * Event transfer (aka. dispatch) is done using \ref event::Channel "event channels".
 * \ref event::Channel "event channels" are callbacks with multiple targets
 * (aka. publishers or signals).
 * An \ref event::Channel "event channel" manages a list of Callback receivers
 * (aka. subscribers, sinks or slots) which are called whenever an
 * event is emitted to the \ref event::Channel "event channel".
 * Callback receiver can be objects of any class that provides handler-methods
 * for one or more events.
 */
namespace event { namespace detail {

struct IdGenerator
{
	static int next();
	static int current();
private:
	static int id_s;
};

template <class To, class From>
To down_cast(From f, To = 0)
{
	assert(dynamic_cast<To>(f) && "event ids are not unique!");
	return static_cast<To>(f);
}

}	// end namespace event::detail

/*!
 * \brief The class Event is used to register events of type T for the use
 * with a \ref event::ChannelManager "ChannelManager".
 *
 * Event associates a unique ID with the template parameter T which is needed
 * by the \ref event::ChannelManager "ChannelManager" in order to create
 * a mapping between event types and their appropriate \ref event::Channel "event channels".
 */
template <class T>
struct Event
{
	static int registerEvent()
	{
		return getId();
	}
  static int	getId() {
#if defined(_MSC_VER) && _MSC_VER <= 1310
    if (id == -1)
      id = detail::IdGenerator::next();
#else
    static int id = detail::IdGenerator::next();  
#endif
    return id;
  }
#if defined(_MSC_VER) && _MSC_VER <= 1310
  static int id;
#endif
};
#if defined(_MSC_VER) && _MSC_VER <= 1310
template <class T>
int Event<T>::id = -1;
#endif


/*!
 * \brief The class ChannelManager manages and provides accss
 * to \ref event::Channel "event channels" for arbitrary event types.
 *
 * ChannelManager creates channels on demand (i.e. the first time they are requested).
 * The lifetime of a Channel created by a ChannelManager is managed by that
 * ChannelManager, i.e. the Channel exists as long as the ChannelManager exists
 * (or ChannelManager::destroyChannelFor is explicitly called).
 *
 * Channels and event types are managed in a 1:1 relationship. That is
 * a ChannelManager manages at most one channel for each different event type.
 *
 */
class ChannelManager
{
public:
	/*!
	 * \brief creates a ChannelManager.
	 */
	ChannelManager();

	/*!
	 * \brief destroys the ChannelManager thereby destroying all managed channels.
	 */
	~ChannelManager();

	/*!
	 * \brief returns a channel that can transfer events of type T.
	 * \see event::Channel
	 */
	template <class T>
	Channel<T>& getChannelFor(event::Event<T>)
	{
		assert(event::Event<T>::getId() >= 0);
		unsigned int id = event::Event<T>::getId();
		if (id >= channels_.size())
			channels_.resize(id + 1);
		if (!channels_[id])
			channels_[id] = new event::Channel<T>();
		return *event::detail::down_cast<event::Channel<T>*>(channels_[id]);
	}

	/*!
	 * \brief const overload for \ref event::ChannelManager::getChannelFor
	 */
	template <class T>
	const Channel<T>& getChannelFor(event::Event<T>) const
	{
		return const_cast<event::ChannelManager*>(this)->getChannelFor(event::Event<T>());
	}

	/*!
	 * \brief dispatches e to the appropriate event channel.
	 *
	 * \see event::Channel::dispatch
	 */
	template <class T>
	void dispatch(const T& e) const
	{
		assert(event::Event<T>::getId() >= 0);
		unsigned int id = event::Event<T>::getId();
		if (id < channels_.size() && channels_[id])
			event::detail::down_cast<event::Channel<T>*>(channels_[id])->dispatch(e);
	}

	/*!
	 * \brief destroys the channel for events of type T
	 * \pre \ref event::Channel::isEventActive is false
	 */
	template <class T>
	void destroyChannelFor(event::Event<T>)
	{
		assert(event::Event<T>::getId() >= 0);
		unsigned int id = event::Event<T>::getId();
		if (id < channels_.size())
		{
			assert(!channels_[id] || !event::detail::down_cast<Channel<T>*>(channels_[id])->isEventActive());
			delete channels_[id];
			channels_[id] = 0;
		}
	}
private:
	ChannelManager(const ChannelManager&);				// ChannelManager
	ChannelManager& operator=(const ChannelManager&);	// does not support copying

	typedef std::vector<detail::ChannelBase*> ChannelList;
	ChannelList channels_;
};

namespace detail {

enum Action {CONNECT, DISCONNECT};

template <class T>
struct Type {};

template <class HandlerType>
inline void forEachInList(event::ChannelManager& m,
						  HandlerType& h,
						  event::detail::Type<event::detail::NoEvent>,
						  event::detail::Action a,
						  int)
{}

template <class HandlerType, class TList>
inline void forEachInList(event::ChannelManager& m,
						  HandlerType& h,
						  event::detail::Type<TList>,
						  event::detail::Action a,
						  double)
{
	typedef typename TList::Head Head;
	typedef typename TList::Tail Tail;
	if (a == CONNECT)
		m.getChannelFor(event::Event<Head>()).connect(h);
	else if (a == DISCONNECT)
		m.getChannelFor(event::Event<Head>()).disconnect(h);
	event::detail::forEachInList(m, h, event::detail::Type<Tail>(), a, 0);
}

}	// end namespace event::detail

/*!
 * \brief The function connectTo connects a Handler to the event channels
 * for the event types in the \ref event::EventList "EventList" EList using a ChannelManager.
 * \param m The ChannelManager that should be used for accessing the event channels
 * \param h The Handler to be connected.
 * \par Usage:
 * \code
 * event::ChannelManager m;
 * Handler h;
 * event::connectTo<event::EventList<Event1, Event2, Event3> >(m, h);
 * \endcode
 * This is equivalent to:
 * \code
 * event::ChannelManager m;
 * Handler h;
 * m.getChannelFor(event::Event<Event1>()).connect(h);
 * m.getChannelFor(event::Event<Event2>()).connect(h);
 * m.getChannelFor(event::Event<Event3>()).connect(h);
 * \endcode
 */
template <class EList, class H>
void connectTo(ChannelManager& m, H& h, EList* = 0)
{
	typedef typename EList::Result TList;
	event::detail::forEachInList(m, h, event::detail::Type<TList>(), detail::CONNECT, 0);
}

/*!
 * \brief The function disconnectFrom disconnects a Handler from the event channels
 * for the event types in the \ref event::EventList "EventList" EList using a ChannelManager.
 * \param m The ChannelManager that should be used for accessing the event channels
 * \param h The Handler to be disconnected.
 * \par Usage:
 * \code
 * event::ChannelManager m;
 * Handler h;
 * event::disconnectFrom<event::EventList<Event1, Event2, Event3> >(m, h);
 * \endcode
 * This is equivalent to:
 * \code
 * event::ChannelManager m;
 * Handler h;
 * m.getChannelFor(event::Event<Event1>()).disconnect(h);
 * m.getChannelFor(event::Event<Event2>()).disconnect(h);
 * m.getChannelFor(event::Event<Event3>()).disconnect(h);
 * \endcode
 */
template <class EList, class H>
void disconnectFrom(ChannelManager& m, H& h, EList* = 0)
{
	typedef typename EList::Result TList;
	event::detail::forEachInList(m, h, event::detail::Type<TList>(), detail::DISCONNECT, 0);
}

}	// end namespace event

#endif
