/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         *
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         *
 ***************************************************************************/
#ifndef LIBNOMORE_GRAPH_H
#define LIBNOMORE_GRAPH_H

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <color.h>
#include <body_node.h>
#include <head_node.h>
#include <stack.h>
#include <event/channel_manager.h>

#include <iterator>
#include <cassert>

namespace NS_NOMORE {

//! Iterator for the body and head node vector skipping nodes.
/*!
 *  The iterator is a template that use a "Skip" class whith decides that a 
 *  node is skipped or not. */
template <class SkipT, class IterT = std::vector<Node*>::const_iterator>
class SkipIterator 
  : public std::iterator<std::forward_iterator_tag, typename SkipT::ResultType> {
public:
  //! Typedef of the iterator of the base class.
  typedef IterT base_iterator;

  //! Typedef of the result type of the iterator.
  typedef typename SkipT::ResultType ResultType;

  //! Initialize the iterator by another iterator.
  /*!
   *  \param begin The begin iterator of the other set to import. 
   *  \param end The end iterator of the other set to import. */
  SkipIterator(base_iterator begin, base_iterator end)
    : begin_(begin)
    , end_(end) {
    skip();
  }

  //! Get the next iterator.
  /*!
   *  \return The iterator of the next position is returned. If no next 
   *          position exists the end() iterator is returned.*/
  SkipIterator& operator++() {
    ++begin_;
    skip();
    return *this;
  }

  //! Get the next iterator.
  /*!
   *  \return The iterator of the next position is returned. If no next 
   *          position exists the end() iterator is returned.*/
  SkipIterator operator++(int) {
    SkipIterator temp(*this);
    ++*this;
    return temp;
  }

  //! Returns the result type of the set.
  /*!
   *  \return The result type set by the "Skip" class. */
  ResultType operator*() const {
    return SkipT::get(*begin_);
  }

  //! Compares two skip iterators for equality.
  /*!
   *  \param other The other skip iterator check for equality.
   *  \return True if both iterators are equal. False otherwise. */
  bool operator==(const SkipIterator& other) const {
    return begin_ == other.begin_;
  }

  //! Compares two skip iterators for equality.
  /*!
   *  \param other The other skip iterator check for equality.
   *  \return False if both iterators are equal. True otherwise. */
  bool operator!=(const SkipIterator& other) const {
    return !(*this == other);
  }

  //! Compares two skip iterators for equality.
  /*!
   *  \param other The other skip iterator check for equality.
   *  \return True if both iterators are equal. False otherwise. */
  bool operator==(const base_iterator& other) const {
    return begin_ == other;
  }

  //! Compares two skip iterators for equality.
  /*!
   *  \param other The other skip iterator check for equality.
   *  \return False if both iterators are equal. True otherwise. */
  bool operator!=(const base_iterator& other) const {
    return !(*this == other);
  }

  //! Returns the end iterator of the set.
  /*!
   *  /return The iterator beyond the last item in the set. */
  base_iterator end() const { 
    return end_; 
  }

private:
  // instance methods
  void skip() {
    while (begin_ != end_ && SkipT::skip(*begin_))
      ++begin_;
  }

  // instance variables
  base_iterator begin_;
  base_iterator end_;
};

//! Struct used for the skip iterator skipped nodes of a specified type.
template <class RT, NodeType::Value type>
struct OnlyNodesOfType {
  //! Typedef of the result type.
  typedef RT ResultType;

  //! Skipping nodes not of type 'type'.
  /*!
   *  \param n The node which should be tested for skipping. 
   *  \return True if the node must be skipped. */
  static bool skip(Node* n) { 
    return n->getType() != type; 
  }

  //! Returns the node of an iterator.
  /*!
   *  \param n The node to get the result type of.
   *  \result The result type of the node n. */
  static ResultType get(Node* n) {
    assert(n->getType() == type);
    return static_cast<ResultType>(n);
  }
};

//! Struct used for the skip iterator skipped all colored nodes.
struct OnlyUncloredNodes {
  //! Typedef of the result type.
  typedef Node* ResultType;

  //! Skipping nodes that are not colored as Color::none.
  /*!
   *  \param n The node which should be tested for skipping. 
   *  \return True if the node must be skipped. */
  static bool skip(Node* n) { 
    return n->getColor() != Color::none; 
  }

  //! Returns the node of an iterator.
  /*!
   *  \param n The node to get the result type of.
   *  \result The result type of the node n. */
  static ResultType get(Node* n) { 
    return n; 
  }
};

class ChoiceOperator;
class Operator;

//! This class implements nomore++'s body head graph. 
/*! 
 *  The main responsibility of a graph is to store the set of head and body 
 *  nodes of a logic program (edges between the nodes are stored in the nodes 
 *  not in  the graph). In addition a graph manages a 
 *  \ref BacktrackStack "backtracking stack" to be used to restore and recolor 
 *  choice points. Furthermore a graph is the central place where event 
 *  handlers register for specific events.
 *
 *	\see _event
 *
 *	\note The Graph should be used as a static graph, i.e. you construct the 
 *  graph in a seperate construction step and once this step is over you call 
 *  the graph's \ref Graph::finishConstruction "finishConstruction" method to 
 *  signal that the graph's structure is now immutable. */
class Graph {
public:
  //! Typedef for the set of nodes in the graph.
  typedef std::vector<Node*> NodeSet;

  //! Typedef for the size type of the set of nodes in the graph.  
  typedef NodeSet::size_type size_type;
  
  //! Forward iterators for iterating the node set (only head nodes).
  typedef SkipIterator<OnlyNodesOfType<HeadNode*, NodeType::head_node> > HeadNodeIterator;
  
  //! Forward iterators for iterating the node set (only body nodes).
  typedef SkipIterator<OnlyNodesOfType<BodyNode*, NodeType::body_node> > BodyNodeIterator;
  
  //! Forward iterators for iterating the node set (only uncolored nodes).
  typedef SkipIterator<OnlyUncloredNodes> UncoloredNodeIterator;

  //! Creates an empty graph and starts its construction phase.
  /*!
   *  \param tightnessGradient The tightness gradient used to compute the 
   *                           tightness factor.
   *
   *  \see HeadNode
   *  \see BodyNode */
  explicit Graph(int tightnessGradient = 3);

  //! Destroys the graph and all contained nodes.
  ~Graph();

  //! Creates and inserts a new body node.
  /*! 
   *  \pre The graph is in the construction phase: isConstructionFinished() 
   *       returns false.
   *
	 *  \return Returns a pointer to the newly created body node.
	 *	\attention The lifetime of the created body node is managed by the graph. */
  BodyNode* insertBodyNode();

  //! Creates and inserts a head node for the atom with the given atomId.
  /*! 
   *  \pre The graph is in the construction phase: isConstructionFinished() 
   *       returns false.
   *
	 *  \return Returns a pointer to a head node with the given id.
	 *	\attention The graph does not check whether a head node for the given 
   *             atomId already exists.
   *  \attention The lifetime of the created head node is managed by the graph. */
  HeadNode* insertHeadNode(long atomId);

  //! Returns the number of head nodes this graph contains.
  /*!
   *  \return The number of head nodes registered in the graph. */
	size_type countHeadNodes() const;
	
	//! Returns the number of body nodes this graph contains.
  /*!
   *  \return The number of body nodes registered in the graph. */
	size_type countBodyNodes() const;

  //! Returns the number of nodes this graph contains.
  /*!
   *  \return The number of nodes (body and head nodes) registered in the 
   *          graph. */
	size_type countNodes() const {
	  return nodes_.size();
  }

  //! Returns the number of rules this graph contains.
  /*!
   *  \note The number of rules is the sum of all body node successors. 
   *
   *  \return The number of rules of the underlying logic program. */
  size_type countRules() const;

  //! Returns an iterator that points to the first head node of this graph.
	/*! 
   *  If this graph does not have any head nodes the end iterator is returned.
   *  
   *  \return The iterator pointed to the first head node.
	 *  \note Traversing the sequence of head nodes yields the nodes in ascending
	 *	      id order. */
	HeadNodeIterator headNodesBegin() const {
    return HeadNodeIterator(nodes_.begin(), nodes_.end());
  }

  //! Returns an iterator that points just beyond the end of head nodes of this graph.
	/*! 
   *  \return The iterator pointed beyond the last head node. */
	HeadNodeIterator headNodesEnd() const {
    return HeadNodeIterator(nodes_.end(), nodes_.end());
  }

  //! Returns an iterator that points to the first body node of this graph.
	/*! 
   *  If this graph does not have any body nodes the end iterator is returned.
   *  
   *  \return The iterator pointed to the first body node.
	 *  \note Traversing the sequence of body nodes yields the nodes in ascending
	 *	      id order. */
	BodyNodeIterator bodyNodesBegin() const {
    return BodyNodeIterator(nodes_.begin(), nodes_.end());
  }

  //! Returns an iterator that points just beyond the end of body nodes of this graph.
	/*! 
   *  \return The iterator pointed beyond the last body node. */
	BodyNodeIterator bodyNodesEnd() const {
    return BodyNodeIterator(nodes_.end(), nodes_.end());
  }

  //! Returns true if the current coloring of the graph is invalid.
  /*!
   *  \return True if the coloring of the graph is invalid. False otherwise. */
  bool hasConflict() const {
    return hasConflict_;
  }
  
  //! Colors a node.
  /*! 
   *  Calls the method Node::setColor() of the node and fires a color event if 
   *  the node's color was changed.
   * 
   *  \pre The current coloring is valid: hasConflict() == false.
   *
   *  \param n The node to be colored.
   *  \param c The new color for the node.
   *  \return Returns true if c is a valid color for n and false otherwise.
   *
	 *	\see Node::setColor()
   *
   *  \post If color() returned true n->getColor() == c. Otherwise an invalid 
   *        coloring is produced: hasConflict() == true. */
  bool color(Node& n, Color::ColorValue c) {
    return color(n, c, Color::none, ColorOpType::no_choice);
  }

  //! Colors a node as choice point.
  /*! 
   *  Calls the method Node::setColor() of the node and fires a color choice 
   *  point event if the node's color was changed.
   *
   *  \pre The current coloring is valid: hasConflict() == false.
   *
   *  \param n    The node to be colored.
   *  \param c    The new color for the node.
   *  \param btc  The choice point's alternative color that is used when the 
   *              choice point is recolored.
   *  \param type Must be either first_choice or next_choice. Used for the
   *              NAryBacktrackingSolver.
   *  \return Returns true if c is a valid color for n and false otherwise.
   *
	 *	\see Node::setColor()
   *
   *  \post If color() returned true n->getColor() == c. Otherwise an invalid 
   *        coloring is produced: hasConflict() == true. */
  bool colorChoicePoint(Node& n, Color::ColorValue c, Color::ColorValue btc, ColorOpType::Value type) {
    assert(n.getColor() == Color::none && (type == ColorOpType::first_choice || type == ColorOpType::next_choice));
    return color(n, c, btc, type);
  }

  //! Colors a node as choice point that can not be recolored.
  /*! 
   *  Calls the method Node::setColor() of the node and fires a color choice 
   *  point event if the node's color was changed. This method colors a stable
   *  choice point. That means you put a normal node onto the stack and fires a 
   *  choice point event because this node (or choice point) is not recolorable.
   *
   *  \pre The current coloring is valid: hasConflict() == false.
   *
   *  \param n The node to be colored.
   *  \param c The new color for the node.
   *  \return Returns true if c is a valid color for n and false otherwise.
   *
	 *	\see Node::setColor()
   *
   *  \post If color() returned true n->getColor() == c. Otherwise an invalid 
   *        coloring is produced: hasConflict() == true. */
  bool colorChoicePointStable(Node& n, Color::ColorValue c) {
    assert(n.getColor() == Color::none);
    return color(n, c, Color::none, ColorOpType::last_choice);
  }

  //! Restores a node.
  /*! 
   *  Calls the method Node::restoreColor() of the node.
   *
   *  \pre c is either Color::none or Color::weak_plus.
   *
   *  \param n The node to be restored.
   *  \param c The new color of the node. 
   *
	 *	\see Node::restoreColor() */
  void restore(Node& n, Color::ColorValue c);
  
	//! Checks if the graph is total colored.
  /*! 
   *  That means all nodes (head and body nodes) are colored 
   *  \ref Color::plus "plus" or \ref Color::minus "minus".
   *
   *  \return True if the graph is total colored else false. */
  bool totalColoring() const {
    return countUncoloredNodes() == 0;
  }

  //! Deletes all nodes of this graph, clears the backtracking stack and starts a new construction phase.
  /*! 
   *  \note Clearing a graph does not change the graph's 
   *        \ref event::ChannelManager "ChannelManager", i.e. connected 
   *        handlers stay connected. */
  void clear();

  
	//! Returns the number of uncolored nodes this graph contains.
  /*! 
   *  \note A node counts as uncolored if its color is either Color::none or 
   *        Color::weak_plus. 
   *  \return The number of nodes that are uncolored. */
	size_type countUncoloredNodes() const {
	  return uncoloredNodes_ + weakColoredNodes_;
  }

  //! Returns the number of weak_plus colored nodes of the graph.
  /*!
   *  \return The number of nodes that are Color::weak_plus. */
  size_type countWeakColoredNodes() const {
    return weakColoredNodes_;
  }

  //! Restores the graph to the state it had just \b after the last choice point was colored.
	/*!
   *  I.e. restores all nodes that were colored after the last choice point 
   *  was colored but not the choice point itself.
   * 
   *  \pre At least one choice point was previously colored.
   *
	 *  \see BacktrackStack::restore()  */
  void restoreToChoicePoint();

  //!	Restores the graph to the state it had just \b before the last choice point was colored.
	/*! 
   *  I.e. restores all nodes that were colored after the last choice point 
   *  was colored as well as the last choice point itself.
   *  
   *  \note The branch that stored the consequences of the last choice point 
   *        is discarded.
   *  \return True if the stack is restored successfully. False if the stack 
   *          is empty. */
  bool restoreChoicePoint();

  //!	First calls restoreLastChoicePoint then recolors the last choice point to its alternative color.	
	/*! 
   *  Fires a color choice point event. If recoloring the last choice points 
   *  leads to an invalid coloring the second last choice point is restored 
   *  and recolored. This process is repeated until either the recoloring of a 
   *  choice point is sucessfull or there are no more choice points left. 
   *  
   *  \return True if the choice point is successfully recolored. False no 
   *          choice point could be restored. */
  bool recolorChoicePoint();

  
  //! Returns an iterator that points to the first uncolored node of this graph.
	/*! 
   *  If this graph does not have any uncolored nodes the end iterator is 
   *  returned.
   *  
   *  \return The iterator pointed to the first uncolored node.
	 *  \note Traversing the sequence of uncolored nodes yields the nodes in ascending
	 *	      id order. */
  UncoloredNodeIterator uncoloredNodesBegin() const {
    return UncoloredNodeIterator(nodes_.begin(), nodes_.end());
  }
  
  //! Returns an iterator that points just beyond the end of uncolored nodes of this graph.
	/*! 
   *  \return The iterator pointed beyond the last uncolored node. */
  UncoloredNodeIterator uncoloredNodesEnd() const {
    return UncoloredNodeIterator(nodes_.end(), nodes_.end());
  }

  //! Returns the node with the given id.
  /*!
   *  \pre 0 <= id < countNodes()
   *  \param id The id used to return the specified node.
   *  \return A pointer to the desired node. */
  Node* getNode(long id) const {
    assert(0 <= id && size_type(id) < countNodes());
    return nodes_[id];
  }
  
  //! Returns the \ref event::ChannelManager "ChannelManager" of this graph.
	/*! 
   *  Interested event handlers need to register with this 
   *  \ref event::ChannelManager "ChannelManager" in order to receive events
	 *  concerning this graph. 
   *
   *  \return The event channel manager of the graph. */
	event::ChannelManager& getEventManager() {
    return eventManager_;
  }

	//! Ends the construction phase and fires a GraphConstructionFinished event.
	/*! 
   *  After calling this method the graph's structure is immutable and calls 
   *  to the insert methods are prohibited. */
	void finishConstruction();

	//! Returns true if the graph's construction is completed.
	/*! 
   *  \return True if the construction phase is over and the graph's structure 
   *          can't change anymore. False if the the graph is now in the 
   *          construction phase and the graph's structure is still volatile. */
	bool isConstructionFinished() const;

  //! Totalizes the \ref NS_NOMORE::Graph "graph".
  /*! 
   *  Totalize colors all uncolored nodes to Color::minus. 
   * 
   *  \pre The current coloring is valid: hasConflict() == false.
   *
   *  \note This operation is reasonable only if no operator is able to color 
   *        the uncolored nodes. For example the 
   *        \ref ForwardPropagator "forward propagator" and
   *        \ref BackwardPropagator "backward propagator" can not always 
   *        decide whether a positve (zero) loop must be colored with 
   *        Color::plus or Color::minus.
   *
   *  \note Totalize was formerly implemented as the post operator N.
   *
   *	\note This operator does not verify if the resulting coloring is valid.
   *        This needs to be done in a seperate propagation step (for example 
   *        using the \ref ForwardPropagator "forward propagator").
   *
   *  \return 
   *    - ColorOperation::success if all uncolored nodes were colored to minus 
   *      or
   *    - ColorOperation::error if the coloring of at least one node would be 
   *      invalid or
   *    - ColorOperation::unchanged if no node was colored. */
  ColorOpResult totalize();

  //! Tightness values for the graph (or logic program).
  enum Tightness {
    //! A graph is tight if all nodes are trivialy connected. 
    tight, 

    //! A graph is tight if there is some strongly connected component which is not trivial (contains more than one node).
    non_tight, 
    
    //! The graph is not tested for tightness.
    indeterminate
  };
  
  //! Returns the tightness state of the graph.
  /*!
   *  \return The tightness state of the graph.
   *
   *  \see Tightness */
  Tightness tightness() const;
  
  //! Checks the graph's tightness after its construction is finished. 
  /*!
   *  \note The check is done only if finishConstruction is called. */
  void checkTightness();

  //! Sets the graph's tightness.
  /*!
   *  \note The value of t is not validated. Use checkTightness to let the 
   *        graph identify its tightness.
   *
   *  \param t The tightness state. */
  void setTightness(Tightness t);

  //! Returns the tightness factor of node n.
  /*!
   *  \param n The node to compute its tightness factor.
   *  \return If the graph is tight (or its tightness is indeterminate) the
   *          method returns 1.0. Otherwise it returns n->getTightnessFactor(). */
  double getTightnessFactor(const Node* n) const {
    if (tight_ == Graph::non_tight && tightnessGradient_ != 0) {
      n->getTightnessFactor(tightnessGradient_);
    }
    return 1.0;
  }

private:
  // deny constructors
  Graph(const Graph& g);
  Graph& operator=(const Graph& g);
  
  // instance methods
  inline void colorChanged(Node& n, Color::ColorValue old);
  bool color(Node& n, Color::ColorValue c, Color::ColorValue btc, ColorOpType::Value);

  // instance variables
  NodeSet nodes_;

  BacktrackStack backtrackingStack_;   //!< Backtracks stack used to restore nodes of this graph.
	event::ChannelManager eventManager_; //!< ChannelManager that manages all event channels needed by this graph and its nodes.
  size_type uncoloredNodes_;
  size_type weakColoredNodes_;
  int tightnessGradient_;
  bool constructionFinished_;	         //!< state of the construction phase.
  bool hasConflict_;
  bool checkTightness_;
  Tightness tight_;
};

//! Output operator for printing the graph to stdout or to a file.
std::ostream& operator<<(std::ostream& os, const Graph& grp);

//! Output operator for printing simple head node to stdout or to a file.
std::ostream& operator<<(std::ostream& os, const HeadNode& head);

//! Output operator for printing simple body node to stdout or to a file.
std::ostream& operator<<(std::ostream& os, const BodyNode& body);

//! Event type fired to signal that the construction of a graph is completed.
/*!  
 *  A graph fires this event when it receives the 
 *  \ref Graph::finishConstruction "finishConstruction" message to signal 
 *  that the graph's structure is now stable and won't change anymore. */
struct GraphConstructionFinished {
  //! Initializes the event.
  /*!
   *  \param theGraph The graph that has finished its construction phase. */
	explicit GraphConstructionFinished(Graph& theGraph)
		: theGraph_(&theGraph) {
	}

  //! The graph fires this event.
	Graph* theGraph_;
};

} // NS_NOMORE

#endif
