/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_HEADNODE_H
#define LIBNOMORE_HEADNODE_H

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <node.h>
#include <vector>
#include <string>
#include <cassert>

namespace NS_NOMORE {

class BodyNode;

//! A head node represents a rule's head of a logic program in nomore++'s body head graph.
/*!
 *  A head node contains zero and one successor body nodes, predecessor body nodes and an
 *  atom name. The successors and predecessors are used to build the graph.
 * 
 *  \see Graph */
class HeadNode : public Node {
public:
  //! Set of body nodes used as predecessors and successors.
  typedef std::vector<BodyNode*> BodyNodeSet;
	
	//! Iterator type that can be used to iterate over a set of body nodes.
	typedef BodyNodeSet::const_iterator BodyNodeIterator;

  //! Constructs a new node representing the atom with the given atomId.
  /*!
   *  \param id     The unique id to be used for the new node.
   *  \param atomId The id of the atom that is represented by this node 
   *                (normally this id comes from lparse). */
  HeadNode(long id, long atomId);

  //! Constructs a new node representing the atom with the given atomId.
  /*!
   *  \param id     The unique id to be used for the new node.
   *  \param atomId The id of the atom that is represented by this node 
   *                (normally this id comes from lparse). 
   *  \param name   The string representation of the node (the name). */
  HeadNode(long id, long atomId, const std::string& name);

  //! Destroys this head node and its members.
	~HeadNode();

  //! Returns the id of the atom represented by this head node.
  /*!
   *  \return The atom id of the head node. */
  long getAtomId() const;
  
  //! Sets the name of the atom represented by this head node.
  /*!
   *  \param name The string representation of the head node. */
  void setAtomName(const std::string& name);  
  
  //! Returns the name of the atom represented by this head node.
  /*!
   *  \return The string representation of the head node. */
  const std::string& getAtomName() const;

  //! Returns an iterator that points to the first predecessor.
	/*! 
   *  If this head node does not have any predecessors the end-iterator is 
   *  returned. 
   *
   *  \return The begin iterator for the predecessors. */
	BodyNodeIterator predecessorsBegin() const {
		return predecessors_.begin();
	}

	//! Returns an iterator that points just beyond the end of the predecessors.
  /*!
   *  \return The end iterator for the predecessors. */
	BodyNodeIterator predecessorsEnd() const {
		return predecessors_.end();
	}
  
	//! Returns an iterator that points to the first zero successor.
	/*! 
   *  If this head node does not have any successors the end iterator is 
   *  returned.
   *
   *  \return The begin iterator for the zero successors. */
	BodyNodeIterator zeroSuccessorsBegin() const {
		return zeroSuccessors_.begin();
	}

	//! Returns an iterator that points just beyond the end of the zero successors.
  /*!
   *  \return The end iterator for the zero successors. */
	BodyNodeIterator zeroSuccessorsEnd() const {
		return zeroSuccessors_.end();
	}

	//! Returns an iterator that points to the first one successor.
	/*! 
   *  If this head node does not have any successors the end iterator is 
   *  returned.
   * 
   *  \return The begin iterator for the one successors. */
	BodyNodeIterator oneSuccessorsBegin() const {
		return oneSuccessors_.begin();
	}

	//! Returns an iterator that points just beyond the end of the one successors.
  /*!
   *  \return The end iterator for the one successors. */
	BodyNodeIterator oneSuccessorsEnd() const {
		return oneSuccessors_.end();
	}

  //! Updates internal counters to reflect the coloring of a predecessor.
  /*! 
   *  \param oldColor The old color of the predecessor.
   *  \param newColor The new color of the predecessor.
   *
   *  \note This method is public for testing reasons only. Normally this 
   *        method should only be called by predecessors of this node. */
  virtual void predecessorColored(Color::ColorValue oldColor, Color::ColorValue newColor);

  //! Updates internal counters to reflect the fact that a predecessor was restored.
  /*! 
   *  \param oldColor The old color of the predecessor.
   *  \param newColor The new color of the predecessor.
   *
   *  \note This method is public for testing reasons only. Normally this 
   *        method should only be called by predecessors of this node. */
  virtual void predecessorRestored(Color::ColorValue oldColor, Color::ColorValue newColor);

  //! Returns the unsupported state of this node.
  /*! 
   *  Unsupported means all predecessors are colored \ref Color::minus "minus".
   *
   *  \return True if the node is unsupported else false. */
  bool isUnsupported() const {
    assert(unsupported_ >= 0);
    return unsupported_ == 0;
  }

	//! Returns the number of predecessors not colored minus.
  /*!
   *  \return The number of predecessors which are not colored Color::minus. */
	long countPredecessorsNotMinus() const {
		return unsupported_;
	}
  
  //! Adds a new predecessor to this node.
  /*! 
   *  \param b The node to be added as predecessor to this node. */
  void insertPredecessor(BodyNode& b);

  //! Adds a zero successor to this node and calls BodyNode::insertZeroPredecessor().
  /*!  
   *  \param b The node to be added as zero successor to this node. 
   *  \post This node was added as a zero predecessor to b. */
  void insertZeroSuccessor(BodyNode& b);

  //! Adds a one successor to this node and calls BodyNode::insertOnePredecessor().
  /*!  
   *  \param b The node to be added as one successor to this node. 
   *  \post This node was added as a one predecessor to b. */
  void insertOneSuccessor(BodyNode& b);

  //! Returns the number of one successors this node has.
  /*!
   *  \return The count of one successors. */
  size_t countOneSuccessors() const {
    return oneSuccessors_.size();
  }

  //! Returns the color which preserves the recursive support.
  /*!
   *  \return Always return Color::weak_plus. */
  virtual Color::ColorValue recursiveSupportColor() const {
    return trivialComponent() ? Color::plus : Color::weak_plus;
  }

  //! Returns the preferred choice point color of this node.
  /*!
   *  \return The preferred color value for the node used for the choice 
   *          point is Color::minus. */
  virtual Color::ColorValue preferredChoicePointColor() const {
    return Color::minus;
  }

  //! Returns the alternative choice point color for this node.
  /*!
   *  \return The alternative color value for the node used for the choice 
   *          point is Color::weak_plus. */
  virtual Color::ColorValue alternativeChoicePointColor() const {
    return trivialComponent() ? Color::plus : Color::weak_plus;
  }

  //! The tightness factor for the node.
  /*!
   *  This factor represents the supported state of the node. The supported 
   *  state depends on the supported state of its uncolored (Color::none) 
   *  predecessors.
   *
   *  \param gradient The gradient used to compute the factor.
   *  \return The tightness factor. */
  virtual double getTightnessFactor(int gradient) const;

protected:
  //! Notifies predecessors and successors of this node about the color change.
  /*! 
   *  Calls for all zero successors the BodyNode::zeroPredecessorColored() method,
   *  for all one successors the BodyNode::onePredecessorColored() method and
   *  for all predecessors the BodyNode::successorColored() method.
   *
   *  \param oldColor The old color before coloring this node. */
  virtual void doSetColor(Color::ColorValue oldColor);

  //! Notifies predecessors and successors of this node about the restoring of a color.
  /*! 
   *  Calls for all zero successors the BodyNode::zeroPredecessorRestored() method,
   *  for all one successors the BodyNode::onePredecessorRestored() method and
   *  for all predecessors the BodyNode::successorRestored() method.
   *
   *  \param oldColor The old color before restoring this node. */
  virtual void doRestoreColor(Color::ColorValue oldColor);

  //! Fires an event of type \ref HeadNodeColored.
  /*! 
   *  \param c The channel manager for firing the events.
   *  \param oldColor The old color of the node before coloring.
   *
   *  \ref _event "More information on events." */
  virtual void fireColorEvent(event::ChannelManager& c, Color::ColorValue oldColor);

  //! Fires an event of type \ref HeadNodeColoredChoicePoint.
  /*! 
   *  \param c The channel manager for firing the events.
   *  \param oldColor The old color of the node before coloring.
   *  \param v The choice point color operation.
   *
   *  \ref _event "More information on events." */
  virtual void fireColorChoicePointEvent(event::ChannelManager& c, Color::ColorValue oldColor, ColorOpType::Value v);

private:
  // instance variable.
  long atomId_;
  std::string atomName_;

  // Unsupported counter initialized by the number of uncolored predecessors.
  /*
   * A node is unsupported if this counter is zero. All predecessors are 
   * colored Color::minus. */
  long unsupported_;

  BodyNodeSet predecessors_;   // Set of predecessors of this node.
  BodyNodeSet zeroSuccessors_; //! Set of zero successors of this node.
  BodyNodeSet oneSuccessors_;  //! Set of one successors of this node.
};

//! Event type used to transfer information about the fact that a head node was colored.
/*! 
 *  \see HeadNode::fireColorEvent */
struct HeadNodeColored {
  //! Initializes the head node event.
  /*!
   *  \param h The head node itself.
   *  \param oldColor The old color of the head node h. */
	HeadNodeColored(HeadNode* h, Color::ColorValue oldColor)
		: node_(h)
		, oldColor_(oldColor) {
	}

	HeadNode*					node_;      /*!< The head node that was colored */
	Color::ColorValue	oldColor_;  /*!< The head node's previous color */
};

//! Event type used to transfer information about the fact that a Head node was colored as choice point
/*! 
 *  \see HeadNode::fireColorChoicePointEvent */
struct HeadNodeColoredChoicePoint {
  //! Initializes the head node cp event.
  /*!
   *  \param h The head node itself.
   *  \param oldColor The old color of the head node h. 
   *  \param type The type of color operation for the choice point. */
	HeadNodeColoredChoicePoint(HeadNode* h, Color::ColorValue oldColor, ColorOpType::Value  type)
		: node_(h)
		, oldColor_(oldColor) 
    , type_(type) {
	}

  HeadNode*					  node_;      /*!< The head node that was colored */
  Color::ColorValue   oldColor_;  /*!< The head node's previous color */
  ColorOpType::Value  type_;      /*!< The type of the coloring */
};

} // NS_NOMORE

#endif
