/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_HEURISTIC_H
#define LIBNOMORE_HEURISTIC_H

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <node.h>
#include <constraint.h>

#include <list>
#include <vector>
#include <cassert>

namespace event {
  template <class E> class Channel;
}

namespace NS_NOMORE {

class BodyNode;
class HeadNode;
class Graph;
class Propagator;

//! A choice consists of a node and two colors for the choice point.
/*! 
 *  The two colors (preferred- and alternative) are used for the search 
 *  algorithm. */
class Choice {
public:
  //! Initializes the choice as empty choice.
  Choice()
    : node_(0) {
  }

  //! Initializes the choice.
  /*! 
   *  \param n The node to be used as a choice. 
	 *  \note Sets preferred- and alternative choice point color to the node's 
   *        preferred and alternative choice point color. */
  Choice(Node* n) 
    : node_(n) {
    if (node_) {
      pref_ = node_->preferredChoicePointColor();
      alt_ = node_->alternativeChoicePointColor();
    }
  }

  //! Initializes the choice.
  /*! 
   *  \param n The node to be used as choice.
   *  \param preferred preferred choice color of this choice.
   *  \param alternative alternative choice color of this choice. */
  Choice(Node* n, Color::ColorValue preferred, Color::ColorValue alternative)
    : node_(n)
    , pref_(preferred)
    , alt_(alternative) {
  }

  //! Returns the node contained in this choice.
  /*!
   *  \return The node which is the choice or 0 if the choice is empty. */
  Node* node() const {return node_;}

  //! Returns the preferred color for the choice.
  Color::ColorValue preferredColor() const {assert(isValid()); return pref_;}

  //! Returns the alternative color for the choice.
  Color::ColorValue alternativeColor() const {assert(isValid()); return alt_;}

  //! Returns the state of the choice.
  /*!
   *  \return True if the choice is not empty else false. */
  bool isValid() const { return node_ != 0; }
  
  //! Returns the state of the choice.
  /*!
   *  \return True if the choice is not empty else false. */
  operator bool() const { return isValid();}

private:
  // instance variables
  Node* node_;
  Color::ColorValue pref_;
  Color::ColorValue alt_;
};

//! The default choice creator creates choices that use the default values.
/*! 
 *  The default color values are given by the node which should be a choice 
 *  point. Simply uses the constructor of Choice with one parameter to set
 *  the default color values. */
struct DefaultChoiceCreator {
  //! Creates a choice from the node n.
  /*!
   *  \param n The node for which a choice should be created. */
  Choice createChoice(Node* n) const {
    return Choice(n);
  }
};

//! Defines the possible states of an activity, e.g. the LookaheadExecution event.
struct ActivityState {

  //! Values of possible state.
  enum Value {
  
    //! the activity is started.
    started = 1,
    
    //! the activity is finished.
    finished = 2,
  };
};

//! Event which is fired whenever a heuristic starts or stops execution.
struct HeuristicExecution {

  //! Initializes the event.
  /*!
   *  \param state Sets the state of the event. */
  explicit HeuristicExecution(ActivityState::Value state) : state_(state) {
	}

  //! The event state.
	ActivityState::Value state_;
};


//! Base class for all heuristics.
/*!
 *  Heuristics guide the selection of choice points. They are used together 
 *  with a \ref ChoiceOperator "choice-operator".
 *
 *  The base class holds the graph and raises the HeuristicExecution event. */
class Heuristic {
public: 
  //! Initializes the base heuristics.
  Heuristic();

  //! Destroys the base heuristics.
  virtual ~Heuristic();
  
  //! Sets the graph on which this heuristic should work.
  /*!
   *  \param g The graph on which the heuristics should work on. */
  void setGraph(Graph& g);
  
  //! Gives the heuristic a chance to see the propagator used.
  /*!
   *  The default implementation does nothing.
   *
   *  \note This method is a hack currently needed only in the Neighborhood 
   *        heuristic.
   *  \todo Find a better place for this method. */
  virtual void setPropagator(Propagator& prop);

  //! Returns the name of the heuristic.
  /*!
   *  \return Returns the name of the heuristics used for example for the 
   *          user interface. */
  virtual const char* getName() const = 0;

  //! Returns the "best" node wrt the concrete heuristics satisfying the constraint.
  /*!
   *  \param c The constraint which must be be satisfied. 
   *  \return The choice containing the "best" node according to a concrete 
   *          heuristics or an empty choice else.
   *
   *  \see Choice::isValid*/
  Choice selectNode(const Constraint& c);

protected:
  //! Returns the graph set by setGraph().
  /*! 
   *  \return The graph on which the heuristic is operating. 
   *  \pre The graph must be set before using setGraph(). */
  Graph& getGraph()  {
    assert(graph_);
    return *graph_;
  }

private:
  //! instance methods
  virtual Choice doSelectNode(const Constraint& c) = 0;
  virtual void doSetGraph();

  // instance variables
  Graph* graph_;
  event::Channel<HeuristicExecution>* channel_;
};

//! The strategy always selects the first node satisfying the constraint.
/*!
 *  The simplest of all heuristic selects the first uncolored node that 
 *  satisfies the given constraint. 
 *
 *  \note This heuristic works for head and body nodes. */
class SelectFirst : public Heuristic {
public:
  //! Initializes the heuristic.
  SelectFirst();

private:
  // instance methods
  Choice doSelectNode(const Constraint& c);
  virtual const char* getName() const;
};

// Define the no heuristic strategy.
typedef SelectFirst NoHeuristic;

//! The event which is used to raise events of heuristic is started and stopped.
/*!
 *  This heuristic event is used to update the heuristic stack for the n-ary 
 *  backtracking search algorithm. The event is only raised in the 
 *  NaryBacktrackingSolver if the currently produced results must be stored 
 *  for the next choice. Only for lookahead heuristics in n-ary backtracking 
 *  search this event is important to save the computed informations for 
 *  later choices. */
struct HeuristicValuesEvent {
  //! The possible action for storing or restoring tables of heuristic infos.
  enum Action {

    //! Store the current table on the stack.
    save,

    //! Restore the last table on the stack.
    restore
  };

  //! Initialize the event.
  /*!
   *  \param action Sets the action of the current event. */
  HeuristicValuesEvent(Action action) : action_(action) {    
  }

  //! The action which should be performed.
  Action action_;
};

//! Returns the name of a heuristics.
/*!
 *  \param T The type of the heuristic (class). */
template <class T>
const char* getName(T*);

//! Returns a new heuristics object used in the object factory.
/*!
 *  \param T The type of the heuristic (class). */
template <class T>
Heuristic* createHeuristic(T*);

}

#endif
