/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_STACK_H
#define LIBNOMORE_STACK_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <color.h>

#include <stack>
#include <vector>
#include <utility>
#include <cassert>

namespace NS_NOMORE {

class Node;
class Graph;

//! Type of backtracking strategy for the solver.
struct BacktrackingMode {
  //! Mode values.
  enum Mode {
    //! The symmetric backtracking strategy.
    /*! 
     *  This backtracking strategy is similar to the smodels backtracking 
     *  search. Represents the backtracking search which produces a binary 
     *  backtracking search space. This strategy uses as choice point one 
     *  node and tries both Color::plus and Color::minus.*/
    symmetric,

    //! The asymetric backtracking strategy.
    /*! 
     *  This backtracking strategy is similar to dlvs backtracking search.
     *  Generates a n-ary backtracking search space. A choice point is a list 
     *  of nodes which is tested in all color combinations of Color::plus and 
     *  Color::minus. */
    asymmetric
  };
};

//! This stack is a stack of branches storing backtrack-information for nodes.
/*! 
 *  A backtracking stack (BacktrackStack) consists of a stack of \ref 
 *  BacktrackStack::Item "item". Each \ref BacktrackStack::Item "item" 
 *  corresponds to a colored node (choice point or simply a propagated node)
 *  and its old color (Color::none or Color::weak_plus). Whenever a node is 
 *  colored an corresponding \ref BacktrackStack::Item "item" is pushed onto 
 *  the stack.
 * 
 *  Backtracking from the last choice point means restoring all items (and 
 *  therefore nodes) on the stack until a choice point 
 *  \ref BacktrackStack::Item "item" is found. */
class BacktrackStack {
public:
  //! Simple item for a node containing backtracking information.
  /*! 
   *  An Item stores the information needed to restore the color of one node 
   *  or choice point. */
  struct Item {
      //! Initializes the item by default as non choice point.
      /*!
       *  \param n The node that was colored.
       *  \param oldColor The node color before the node was colored.
       *  \param cp Type of node coloring: Choice point? */
  		Item(Node& n, Color::ColorValue oldColor, bool cp = false)
  			: node_(&n)
  			, color_(oldColor)
        , cp_(cp) {
      }

      //! The node for which this item stores bt-information.
      Node *node_;

      //! The old color of the node node_.
      Color::ColorValue color_;

      //! The type of color action: Choice point?
      bool cp_;
	};
  
  //! Creates an empty backtrack stack.
  BacktrackStack();

  //! Returns true if this backtrack stack is empty.
  bool isEmpty() const {
    return stack_.empty();
  }
  
  //! Pushes a new item representing a simple node onto the stack.
  /*!
   *  \param n Node that was colored and that should be backtrackable.
   *  \param oldColor The node's color before the coloring. 
	 *	\note Only nodes that were colored after the first choice point are 
   *        backtrackable. */
  void pushItem(Node& n, Color::ColorValue oldColor) {
    if (!isEmpty()) 
      stack_.push(Item(n, oldColor));
  }

  //! Pushes a new item representing a choice point onto the backtracking stack.
  /*! 
   *  \param n The node that should be backtrackable.
   *  \param altColor The alternative color for the next backtracking branch. */
  void pushBranch(Node& n, Color::ColorValue altColor) {
    stack_.push(Item(n, altColor, true));
	}

  //! Restores all items (nodes) until a choice point item (node) is found.
  /*! 
   *  \pre The stack is not empty (isEmpty() returns false).
   *  \param g The graph used to restore the nodes.
   *  \post The found choice point item is on the top of the stack. */
  void restore(Graph& g);

  //! Restores all items (nodes) and returns the choice point item.
  /*! 
   *  \pre The stack is not empty (isEmpty() returns false).
   *  \param g The graph used the restore the nodes.
   *  \return An Item containing the alternative color of the restored choice 
   *          point item and the choice point node itself as 
   *          \ref BacktrackStack::Item "item". */
  Item restoreAndPop(Graph& g);
  
private:
  // deny usage
  BacktrackStack(const BacktrackStack&);
	BacktrackStack& operator=(const BacktrackStack&);

  // typedefs
  typedef std::stack<Item, std::vector<Item> > TheStack;

  // instance variables
  TheStack stack_;
};

} // NS_NOMORE

#endif
