/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         *
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         *
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#define for if (0); else for
#endif

#include <graph.h>
#include "detail/functors.h"
#include <algorithm>
#include <choice_operator.h>
#include <strong_components.h>
namespace NS_NOMORE {


Graph::Graph(int tg)
  : uncoloredNodes_(0)
  , weakColoredNodes_(0)
  , tightnessGradient_(tg)
  , constructionFinished_(false)
  , hasConflict_(false)
  , checkTightness_(false) 
  , tight_(indeterminate) {
}

Graph::~Graph() {
  clear();
}

void Graph::finishConstruction() {
	if (!constructionFinished_) {
    constructionFinished_ = true;
    // important: first check for tightness then dispatch event
    // -> ops may depend on tightness-check
    if (checkTightness_ && tight_ == indeterminate) {
      tight_ = identifyStrongComponents(*this) == 0 ? tight : non_tight;
    }
		GraphConstructionFinished e(*this);
		eventManager_.dispatch(e);
	}
}

bool Graph::isConstructionFinished() const {
	return constructionFinished_;
}

BodyNode* Graph::insertBodyNode() {
  assert(!isConstructionFinished());
  BodyNode* nb = new BodyNode(static_cast<long>(countNodes()));
  nodes_.push_back(nb);
  ++uncoloredNodes_;
  return nb;
}

HeadNode* Graph::insertHeadNode(long atomId) {
  assert(!isConstructionFinished());
	assert(atomId > 0);
  HeadNode* node = new HeadNode(static_cast<long>(countNodes()), atomId);
  nodes_.push_back(node);
  ++uncoloredNodes_;
  return node;
}

void Graph::clear() {
  std::for_each(nodes_.begin(), nodes_.end(), DETAIL::DeleteObject());
  nodes_.clear();
	constructionFinished_ = false;
  uncoloredNodes_ = 0;
  weakColoredNodes_ = 0;
  tight_ = indeterminate;
}



Graph::size_type Graph::countRules() const {
  Graph::size_type rules = 0;
  for (BodyNodeIterator it = bodyNodesBegin(); it != it.end(); ++it) {
    rules += std::distance((*it)->successorsBegin(), (*it)->successorsEnd());
  }
  return rules;
}

void Graph::restoreToChoicePoint() {
  backtrackingStack_.restore(*this);
}

void Graph::restore(Node& n, Color::ColorValue c) {
  Color::ColorValue old = n.getColor();
  n.restoreColor(c);
  colorChanged(n, old);
}

bool Graph::color(Node& n, Color::ColorValue c, Color::ColorValue btc, ColorOpType::Value op) {
  assert( hasConflict() == false );
  Color::ColorValue oldColor = n.getColor();
  ColorOpResult colorOp = n.setColor(c);
  if (colorOp == ColorOperation::succeeded) {
    colorChanged(n, oldColor);
    if (op == ColorOpType::no_choice) {
      backtrackingStack_.pushItem(n, oldColor);
      n.fireColorEvent(eventManager_, oldColor);
    }
    else if (op == ColorOpType::first_choice || op == ColorOpType::next_choice) {
      backtrackingStack_.pushBranch(n, btc);
      n.fireColorChoicePointEvent(eventManager_, oldColor, op);
    }
    else if (op == ColorOpType::last_choice) {
      backtrackingStack_.pushItem(n, oldColor);
      n.fireColorChoicePointEvent(eventManager_, oldColor, op);
    }
  }
  else if (colorOp == ColorOperation::failed) {
    hasConflict_ = true;
  }
  return !hasConflict_;
}


bool Graph::recolorChoicePoint() {
  while(!backtrackingStack_.isEmpty()) {
    BacktrackStack::Item cp = backtrackingStack_.restoreAndPop(*this);
    hasConflict_ = false;
    if (colorChoicePointStable(*cp.node_, cp.color_)) {
      return true;
    }
  } // end while
  return false;
}

bool Graph::restoreChoicePoint() {

  if (!backtrackingStack_.isEmpty()) {
    backtrackingStack_.restoreAndPop(*this);
    hasConflict_ = false;
    return true;
  }
  return false;
}

std::ostream& operator<<(std::ostream& os, const Graph& grp) {
  os << "HeadNodes [Count=" << (unsigned int)grp.countHeadNodes() << "]" << std::endl;
  for(Graph::HeadNodeIterator it = grp.headNodesBegin(); it != grp.headNodesEnd(); ++it) {
    HeadNode *head = *it;
    os << "  " << *head << " ["<<head->getColor() <<" - headid="<<head->getId()<<"]" <<std::endl;

    //os << "    Predecessors" << std::endl;
    //for(HeadNode::BodyNodeIterator itBody = head->predecessorsBegin();
    //  itBody != head->predecessorsEnd(); ++itBody) {

    //  os << "      " << *(*itBody) << std::endl;
    //}

    //os << "    0-Successors" << std::endl;
    //for(HeadNode::BodyNodeIterator itBody = head->zeroSuccessorsBegin();
    //  itBody != head->zeroSuccessorsEnd(); ++itBody) {

    //  os << "      " << *(*itBody) << std::endl;
    //}

    //os << "    1-Successors" << std::endl;
    //for(HeadNode::BodyNodeIterator itBody = head->oneSuccessorsBegin();
    //  itBody != head->oneSuccessorsEnd(); ++itBody) {

    //  os << "      " << *(*itBody) << std::endl;
    //}
  }

  os << "BodyNodes [Count=" << (unsigned int)grp.countBodyNodes() << "]" << std::endl;
  for(Graph::BodyNodeIterator it = grp.bodyNodesBegin(); it != grp.bodyNodesEnd(); ++it) {
    BodyNode *body = *it;
    os << "  " << *body << " ["<<body->getColor() <<" - bodyid="<<body->getId()<<"]" <<std::endl;

    //os << "    Successors" << std::endl;
    //for(BodyNode::HeadNodeIterator itHead = body->successorsBegin();
    //  itHead != body->successorsEnd(); ++itHead) {

    //  os << "      " << *(*itHead) << std::endl;
    //}

    //os << "    0-Predecessors" << std::endl;
    //for(BodyNode::HeadNodeIterator itHead = body->zeroPredecessorsBegin();
    //  itHead != body->zeroPredecessorsEnd(); ++itHead) {

    //  os << "      " << *(*itHead) << std::endl;
    //}

    //os << "    1-Predecessors" << std::endl;
    //for(BodyNode::HeadNodeIterator itHead = body->onePredecessorsBegin();
    //  itHead != body->onePredecessorsEnd(); ++itHead) {

    //  os << "      " << *(*itHead) << std::endl;
    //}
  }

  //os << "corresponding Rules " << std::endl;
  //for(Graph::BodyNodeIterator it = grp.bodyNodesBegin(); it != grp.bodyNodesEnd(); ++it) {
  //  BodyNode *body = *it;

  //  for(BodyNode::HeadNodeIterator itHead = body->successorsBegin();
  //    itHead != body->successorsEnd(); ++itHead) {

  //    os << "  " << *(*itHead) << " :- " << *body << "." << std::endl;
  //  }
  //}

  return os;
}

void Graph::colorChanged(Node& n, Color::ColorValue oldColor) {
  Color::ColorValue newColor = n.getColor();
  assert(oldColor != newColor);
  if (oldColor == Color::none)
    --uncoloredNodes_;
  else if (newColor == Color::none)
    ++uncoloredNodes_;
  weakColoredNodes_ += (newColor == Color::weak_plus) - (oldColor == Color::weak_plus);
}


std::ostream& operator<<(std::ostream& os, const HeadNode& head) {
  os << head.getAtomName() << " {" << head.getComponentNumber() << "}";
  return os;
}

std::ostream& operator<<(std::ostream& os, const BodyNode& body) {

  int count = 0;
  for(BodyNode::HeadNodeIterator it = body.zeroPredecessorsBegin();
    it != body.zeroPredecessorsEnd(); ++it) {

    os << *(*it) << ", ";
    count++;
  }

  for(BodyNode::HeadNodeIterator it = body.onePredecessorsBegin();
    it != body.onePredecessorsEnd(); ++it) {

    os << "not " << *(*it) << ", ";
    count++;
  }
  os << " {" << body.getComponentNumber() << "}";

  // only for bodies without preconditions
  if(count == 0)
    os << "{}";

  return os;
}

ColorOpResult Graph::totalize() {
  if (weakColoredNodes_) {
    hasConflict_ = true;
    return ColorOperation::failed;
  }
  if (countUncoloredNodes()) {
    for (UncoloredNodeIterator it = uncoloredNodesBegin(); it != it.end(); ++it) {
      if (!color(**it, Color::minus))
        return ColorOperation::failed;
    }
    return ColorOperation::succeeded;
  }
  return ColorOperation::unchanged;
}

Graph::size_type Graph::countHeadNodes() const {
  return std::distance(headNodesBegin(), headNodesEnd());
}

Graph::size_type Graph::countBodyNodes() const {
  return std::distance(bodyNodesBegin(), bodyNodesEnd());
}

Graph::Tightness Graph::tightness() const {
  return tight_;
}

void Graph::setTightness(Tightness t) {
  tight_ = t;
}

void Graph::checkTightness() {
  checkTightness_ = true;
}



} // NS_NOMORE
