/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#include <operators/default_local_propagator.h>
#include <graph.h>
namespace NS_NOMORE {

DefaultLocalPropagator::DefaultLocalPropagator(Graph& g) 
	: Propagator(g)
	, inConstruction_(true) {

	event::ChannelManager& m = getGraph().getEventManager();
	m.getChannelFor(event::Event<HeadNodeColored>()).connect(*this);
	m.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);
	m.getChannelFor(event::Event<BodyNodeColoredChoicePoint>()).connect(*this);
	m.getChannelFor(event::Event<HeadNodeColoredChoicePoint>()).connect(*this);
	m.getChannelFor(event::Event<GraphConstructionFinished>()).connect(*this);
}

void DefaultLocalPropagator::reset() {
	while ( !bodiesTodo_.empty() ) bodiesTodo_.pop();
  while ( !headsTodo_.empty() ) headsTodo_.pop();
}

bool DefaultLocalPropagator::propagateLocal() {
	assert( !getGraph().hasConflict() );
  bool result = true;
	do {
		if ( (result = propagate(bodiesTodo_)) == true) {
			result = propagate(headsTodo_);
		}
	} while (!bodiesTodo_.empty() && result);
	return result;
}

bool DefaultLocalPropagator::propagateLocal(int steps) {
	assert(steps > 0);
	bool error = false;
	HeadsTodo heads;
	BodiesTodo bodies;
	
	// The idea behind this loop is to store the nodes to be propagated and the results
	// of the propagation in two different queues-sets.
	// This way we can stop propagation early (after steps steps). In normal propagation
	// results are appended to the todo-queues which leads to fixpoint-propagation.
	heads.swap(headsTodo_);
	bodies.swap(bodiesTodo_);	
	for (int i = 0; i < steps; ++i) {
		if (!propagate(heads) || !propagate(bodies)) {
			error = true;
			break;
		}
		// At this point the queues heads and bodies are empty, while headsTodo_ and bodiesTodo_
		// contain the results of the previous propagation step.
		// By swapping the queues we'll propagate the results of step n in step n + 1, while
		// storing the results of step n + 1 in the todo-queues for step n + 2.
		heads.swap(headsTodo_);
		bodies.swap(bodiesTodo_);
	}
	
	// only reset our todo-Queues.
	DefaultLocalPropagator::reset();
	return !error;
}
///////////////////////////////////////////////////////////////////////////////
// Event Handler
///////////////////////////////////////////////////////////////////////////////
void DefaultLocalPropagator::handle(const GraphConstructionFinished& e) {
	assert(e.theGraph_ == &getGraph());
	inConstruction_ = false;
}

void DefaultLocalPropagator::handle(const HeadNodeColored& e) {
	headsTodo_.push( HeadNodePair(e.node_, inConstruction_) );
}

void DefaultLocalPropagator::handle(const BodyNodeColored& e) {
	bodiesTodo_.push( BodyNodePair(e.node_, false) );
}

// Choice points are always backpropagated!
void DefaultLocalPropagator::handle(const HeadNodeColoredChoicePoint& e) {
	headsTodo_.push( HeadNodePair(e.node_, true) );
}

void DefaultLocalPropagator::handle(const BodyNodeColoredChoicePoint& e) {
	bodiesTodo_.push( BodyNodePair(e.node_, true) );
}

///////////////////////////////////////////////////////////////////////////////
// forward propagation
///////////////////////////////////////////////////////////////////////////////
bool DefaultLocalPropagator::forwardProp(const HeadNode& h) {
	Graph& g = getGraph();
	HeadNode::BodyNodeIterator otherSuccsBegin;
	HeadNode::BodyNodeIterator otherSuccsEnd;
	HeadNode::BodyNodeIterator minusSuccsBegin;
	HeadNode::BodyNodeIterator minusSuccsEnd;

	if (HasColor(Color::plus | Color::weak_plus)(&h)) {
		otherSuccsBegin = h.zeroSuccessorsBegin();
		otherSuccsEnd   = h.zeroSuccessorsEnd();
		// all one succs are blocked
		minusSuccsBegin = h.oneSuccessorsBegin();
		minusSuccsEnd   = h.oneSuccessorsEnd();		
	}
	else if (h.getColor() == Color::minus) {
		otherSuccsBegin = h.oneSuccessorsBegin();
		otherSuccsEnd   = h.oneSuccessorsEnd();
		// all zero succs are unsupported
		minusSuccsBegin = h.zeroSuccessorsBegin();
		minusSuccsEnd   = h.zeroSuccessorsEnd();
	}
	else {
		return true; 
	}
	for (; minusSuccsBegin != minusSuccsEnd; ++minusSuccsBegin) {
		if (!g.color(**minusSuccsBegin, Color::minus)) {
			return false;
		}
	}
	for (; otherSuccsBegin != otherSuccsEnd; ++otherSuccsBegin) {
		if( (*otherSuccsBegin)->isWeakSupported() && (*otherSuccsBegin)->isUnblocked() ) {
			if (!g.color(**otherSuccsBegin, (*otherSuccsBegin)->recursiveSupportColor())) {
				return false;
			}
    }
	}
	return true;
}

bool DefaultLocalPropagator::forwardProp(const BodyNode& b) {
	Graph& g = getGraph();
	BodyNode::HeadNodeIterator it = b.successorsBegin();
	BodyNode::HeadNodeIterator end = b.successorsEnd();
	if (HasColor(Color::plus | Color::weak_plus)(&b)) {
		for (; it != end; ++it) {
			if (!g.color(**it, b.getColor())) {
				return false;
			}
		}
	}
	else if (b.getColor() == Color::minus) {
		for (; it != end; ++it) {
			if ( (*it)->isUnsupported() && !g.color(**it, Color::minus) ) {
				return false;
			}
		}
	}
	return true;
}

///////////////////////////////////////////////////////////////////////////////
// backward propagation
///////////////////////////////////////////////////////////////////////////////
bool DefaultLocalPropagator::backwardProp(const HeadNode& h) {
	if (HasColor(Color::plus | Color::weak_plus)(&h)
		&& h.countPredecessorsNotMinus() == 1) {

		HeadNode::BodyNodeIterator it = std::find_if(h.predecessorsBegin(),
			h.predecessorsEnd(), HasColor(Color::none));
		if (it != h.predecessorsEnd()) {
			if (!getGraph().color(**it, (*it)->recursiveSupportColor()))
        return false;
			bodiesTodo_.back().second = true; // backpropagate the colored node
		}

	}
	else if (h.getColor() == Color::minus) {

		HeadNode::BodyNodeIterator end = h.predecessorsEnd();
		for (HeadNode::BodyNodeIterator it = h.predecessorsBegin(); it != end; ++it) {
			if (HasColor(Color::none)(*it)) {
				if (!getGraph().color(**it, Color::minus))
          return false;
				bodiesTodo_.back().second = true; // backpropagate the colored node
			}
		}
	}
  return true;
}

bool DefaultLocalPropagator::backwardProp(const BodyNode& b) {
	if (HasColor(Color::plus | Color::weak_plus)(&b)) {
    BodyNode::HeadNodeIterator end = b.zeroPredecessorsEnd();
		for (BodyNode::HeadNodeIterator it = b.zeroPredecessorsBegin(); it != end; ++it) {
			if ( (*it)->getColor() == Color::none) {
				if (!getGraph().color(**it, (*it)->recursiveSupportColor()))
          return false;
				headsTodo_.back().second = true; // backpropagate the colored node
			}
		}
		end = b.onePredecessorsEnd();
		for (BodyNode::HeadNodeIterator it = b.onePredecessorsBegin(); it != end; ++it) {
			if ( (*it)->getColor() == Color::none) {
				if (!getGraph().color(**it, Color::minus))
          return false;
				headsTodo_.back().second = true; // backpropagate the colored node
			}
		}
	}
	else if (b.getColor() == Color::minus) {
		if (b.isUnblocked() && b.countZeroPredecessorsNotPlusOrWeakPlus() == 1) {
			BodyNode::HeadNodeIterator it = std::find_if(b.zeroPredecessorsBegin(),
				b.zeroPredecessorsEnd(), HasColor(Color::none));
			if (it != b.zeroPredecessorsEnd()) {
				if (!getGraph().color(**it, Color::minus))
          return false;
				headsTodo_.back().second = true; // backpropagate the colored node
			}
		}
		else if (b.isWeakSupported() && b.countOnePredecessorsNotMinus() == 1) {
			BodyNode::HeadNodeIterator it = std::find_if(b.onePredecessorsBegin(),
				b.onePredecessorsEnd(), HasColor(Color::none));
			if (it != b.onePredecessorsEnd()) {
				if (!getGraph().color(**it, (*it)->recursiveSupportColor()))
          return false;
				headsTodo_.back().second = true; // backpropagate the colored node
			}
		}
	}
  return true;
}

bool DefaultLocalPropagator::backwardPropSuccs(const HeadNode& h) {
	// backpropagate head successors -> jumping
	HeadNode::BodyNodeIterator it, end;
	Color::ColorValue newColor = h.getColor();
	if (newColor == Color::plus || newColor == Color::weak_plus) {
		it = h.zeroSuccessorsBegin();
		end = h.zeroSuccessorsEnd();
		
	}
	else if (newColor == Color::minus) {
		it = h.oneSuccessorsBegin();
		end = h.oneSuccessorsEnd();
	}
	else {
		return true;
	}
	for (; it != end; ++it) {
		if ((*it)->getColor() == Color::minus && !backwardProp(**it)) {
			return false;
		}
	}
	return true;
}

bool DefaultLocalPropagator::backwardPropSuccs(const BodyNode& b) {	
	if (b.getColor() == Color::minus) {
		BodyNode::HeadNodeIterator end = b.successorsEnd();
		HasColor hasColorPlus(Color::plus | Color::weak_plus);
		for (BodyNode::HeadNodeIterator it = b.successorsBegin(); it != end; ++it) {
			if (hasColorPlus(*it) && !backwardProp(**it)) {
				return false;
			}
		}
	}
	return true;
}

}

