/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif
#include <operators/imp_unfounded_set.h>
#include <graph.h>
#include <iostream>

#ifdef KDEVELOP
  #include "algorithms.h"
#else
  #include "../detail/algorithms.h"
#endif

#if defined (_MSC_VER) && _MSC_VER <= 1300
#define for if(0); else for
#endif

namespace NS_NOMORE {

///////////////////////////////////////////////////////////////////////////////
template <>
const char* getName(ImprovedUnfoundedSet*) {
  return "PBIU";
}
template <>
Propagator* createPropagator(Graph& g, ImprovedUnfoundedSet*) {
	return new ImprovedUnfoundedSet(g);
}

///////////////////////////////////////////////////////////////////////////////
ImprovedUnfoundedSet::ImprovedUnfoundedSet(Graph& grp) 
	: DefaultLocalPropagator(grp)
  , disabled_(false)
  , activeComponent_(trivialComponent) {
  
	// the base class already registered us for all necessary events.	
	grp.checkTightness();  
}

///////////////////////////////////////////////////////////////////////////////
// common interface
///////////////////////////////////////////////////////////////////////////////
const char* ImprovedUnfoundedSet::getName() const {
  return NS_NOMORE::getName<ImprovedUnfoundedSet>(0);
}

bool ImprovedUnfoundedSet::propagateFull() {
	bool error = propagateLocal() == false;
  
  if (!disabled_) {
		for (HeadNode* head = 0; !todo_.empty() && !error; ) {
      head = todo_.top();
      todo_.pop();      
      if (Color::isUncolored(head->getColor()) && findUnfoundedSet(head)) {
        // we found an unfounded set, now propagate it.        
				error = getGraph().hasConflict() || propagateLocal() == false;
        clearSource();
			}
    }
    clearInternal();
  }  
  assert( error ==  getGraph().hasConflict() );
  return !error;
}


void ImprovedUnfoundedSet::reset() {
  clearInternal();
  DefaultLocalPropagator::reset();
}

void ImprovedUnfoundedSet::clearInternal() {
  while (!todo_.empty()) {
    todo_.pop();
  }
  set_.clear();
  external_.clear();
  clearSource();
}

void ImprovedUnfoundedSet::clearSource() {
  
  source_.assign(source_.size(), 0);
  
}

bool ImprovedUnfoundedSet::isDisabled() const {
  return disabled_;
}

///////////////////////////////////////////////////////////////////////////////
// event handlers
///////////////////////////////////////////////////////////////////////////////
void ImprovedUnfoundedSet::handle(const GraphConstructionFinished& e) {
  assert(e.theGraph_ == &getGraph());
	disabled_ = e.theGraph_->tightness() == Graph::tight;
  if (!disabled_) {
    source_.resize(e.theGraph_->countNodes(), 0);
  }
  DefaultLocalPropagator::handle(e);
}

void ImprovedUnfoundedSet::handle(const BodyNodeColored& e) {
  DefaultLocalPropagator::handle(e);
	if(!disabled_ && e.node_->getColor() == Color::minus) {
    updateTodoList(e.node_);
  }
}

void ImprovedUnfoundedSet::handle(const BodyNodeColoredChoicePoint& e) {
  DefaultLocalPropagator::handle(e);
	if(!disabled_ && e.node_->getColor() == Color::minus) {
    updateTodoList(e.node_);
  }
	
}
///////////////////////////////////////////////////////////////////////////////
// private helpers
///////////////////////////////////////////////////////////////////////////////

bool ImprovedUnfoundedSet::findUnfoundedSet(HeadNode* head) {
  // init sets
  set_.clear();
  external_.clear();

  scc_ = head->getComponentNumber();
  if (scc_ != activeComponent_) {
    // we are done with the active component...
    // start a new one
    clearSource();
    activeComponent_ = scc_;
  }
  set_.insert(head);
  HeadNode::BodyNodeIterator it = head->predecessorsBegin();
  HeadNode::BodyNodeIterator end = head->predecessorsEnd();
  for(; it != end; ++it) {
    if((*it)->getComponentNumber() != scc_ && Color::isUncolored((*it)->getColor()))
      return false;
    
    if((*it)->getComponentNumber() == scc_ && Color::isUncolored((*it)->getColor()))
      external_.push_back(*it);
  }

  checkUnfoundedSet();

  if(set_.empty())
    return false;
  
  color(set_);
  return true;
}

void ImprovedUnfoundedSet::checkUnfoundedSet() {
  
  HeadSet remove;
  BodySet justified;

  while(!external_.empty()) {
    // get first external body
    BodyNode *b = external_.front();
    external_.pop_front();

    HeadNode *a = selectPredOf(b);
    if(a != 0) {
      if(0 == probJustifiedBodies(a)) {
        set_.insert(a);
        updateExternalList(a);        
      } else {
        addToSource(a);
        external_.push_front(b);
      }
    } else {
      addToSource(b);
      getRemovedAtoms(b, remove);

      while(!remove.empty()) {
        HeadSet::iterator it = remove.begin();
        HeadSet::iterator end = remove.end();
        for(; it != end; ++it) {
          set_.erase(*it);
        }

        justified.clear();
        genJustifiedBodies(remove, justified);
        BodySet::iterator bit = justified.begin();
        BodySet::iterator bend = justified.end();
        for(; bit != bend; ++bit) {
          addToSource(*bit);
        }

        getRemovedAtoms(justified, remove);
      }

      updateCompleteExternal();
    }
  }
}

void ImprovedUnfoundedSet::updateExternalList(HeadNode *a) {
  // delete body nodes which positivly depend on a
  HeadNode::BodyNodeIterator it = a->zeroSuccessorsBegin();
  HeadNode::BodyNodeIterator end = a->zeroSuccessorsEnd();
  for(; it != end; ++it) {
    ExternalList::iterator f = std::find(external_.begin(), external_.end(), *it);
    if(f != external_.end())
      external_.erase(f);
  }        

  // insert body nodes which causes a but not depend circularyly on a
  it = a->predecessorsBegin();
  end = a->predecessorsEnd();
  for(; it != end; ++it) {
    if((*it)->getColor() != Color::minus && !circDepended(*it))
      external_.push_back(*it);
  }
}

void ImprovedUnfoundedSet::updateCompleteExternal() {
  external_.clear();
  HeadSet::iterator it = set_.begin();
  HeadSet::iterator end = set_.end();
  for(; it != end; ++it) {
    HeadNode::BodyNodeIterator bit = (*it)->predecessorsBegin();
    HeadNode::BodyNodeIterator bend = (*it)->predecessorsEnd();
    for(; bit != bend; ++bit) {
      if((*bit)->getColor() != Color::minus && !circDepended(*bit))
        external_.push_back(*bit);
    }
  }
}

HeadNode* ImprovedUnfoundedSet::selectPredOf(BodyNode *b) {
  BodyNode::HeadNodeIterator it = b->zeroPredecessorsBegin();
  BodyNode::HeadNodeIterator end = b->zeroPredecessorsEnd();
  for(; it != end; ++it) {
    if(scc_ == (*it)->getComponentNumber() && (*it)->getColor() != Color::plus && !inSink(*it))
      return *it;
  }
  return 0;
}

int ImprovedUnfoundedSet::probJustifiedBodies(HeadNode *a) {
  HeadNode::BodyNodeIterator it = a->predecessorsBegin();
  HeadNode::BodyNodeIterator end = a->predecessorsEnd();
  for(; it != end; ++it) {
    if((*it)->getColor() != Color::minus && ((*it)->getComponentNumber() != scc_ || inSource(*it)))
      return 1;
  }

  return 0;
}

bool ImprovedUnfoundedSet::circDepended(BodyNode* b) {
  BodyNode::HeadNodeIterator it = b->zeroPredecessorsBegin();
  BodyNode::HeadNodeIterator end = b->zeroPredecessorsEnd();
  for(; it != end; ++it) {
    if(inSet(*it))
      return true;
  }

  return false;
}

void ImprovedUnfoundedSet::getRemovedAtoms(BodyNode *b, ImprovedUnfoundedSet::HeadSet &remove) {
  remove.clear();
  BodyNode::HeadNodeIterator it = b->successorsBegin();
  BodyNode::HeadNodeIterator end = b->successorsEnd();
  for(; it != end; ++it) {
    if(inSet(*it)) {
      remove.insert(*it);
      addToSource(*it);
    }
  }
}

void ImprovedUnfoundedSet::getRemovedAtoms(const ImprovedUnfoundedSet::BodySet& justified, ImprovedUnfoundedSet::HeadSet &remove) {
  remove.clear();
  HeadSet::iterator it = set_.begin();
  HeadSet::iterator end = set_.end();
  bool add;
  for(; it != end; ++it) {
    HeadNode::BodyNodeIterator bit = (*it)->predecessorsBegin();
    HeadNode::BodyNodeIterator bend = (*it)->predecessorsEnd();
    add = false;
    for(; bit != bend && !add; ++bit) {
      if(inBodySet(justified, *bit))
        add = true;
    }    
    if(add){
      remove.insert(*it);
      addToSource(*it);
    }
  }
}

void ImprovedUnfoundedSet::genJustifiedBodies(const ImprovedUnfoundedSet::HeadSet& remove, ImprovedUnfoundedSet::BodySet& justified) {
  HeadSet::const_iterator it = remove.begin();
  HeadSet::const_iterator end = remove.end();
  bool add;
  for(; it != end; ++it) {
    HeadNode::BodyNodeIterator bit = (*it)->zeroSuccessorsBegin();
    HeadNode::BodyNodeIterator bend = (*it)->zeroSuccessorsEnd();
    for(; bit != bend; ++bit) {      
      if((*bit)->getComponentNumber() == scc_ && (*bit)->getColor() != Color::minus) {
        add = true;
        BodyNode::HeadNodeIterator hit = (*bit)->zeroPredecessorsBegin();
        BodyNode::HeadNodeIterator hend = (*bit)->zeroPredecessorsEnd();
        for(; hit != hend; ++hit) {
          if((*hit)->getComponentNumber() == scc_ && (*hit)->getColor() == Color::plus && !inSink(*hit))
            add = false;
        }

        if(add) {
          justified.insert(*bit);
        }
      }
    }    
  }
}

bool ImprovedUnfoundedSet::inSink(HeadNode *h) {
  return source_[h->getId()] != 0;
}

bool ImprovedUnfoundedSet::inSet(HeadNode *h) {
  return set_.find(h) != set_.end();
}

bool ImprovedUnfoundedSet::inSource(BodyNode *b) {
  return source_[b->getId()] != 0;
}

bool ImprovedUnfoundedSet::inBodySet(const BodySet& s, BodyNode* b) {
  return s.find(b) != s.end();
}

void ImprovedUnfoundedSet::color(const HeadSet& uset) {
  Graph& g = getGraph();
  for(HeadSet::const_iterator it = uset.begin(); it != uset.end(); ++it) {
    if (!g.color(**it, Color::minus))
      return;      
  }
}

void ImprovedUnfoundedSet::updateTodoList(BodyNode* n) {
  BodyNode::HeadNodeIterator it, end = n->successorsEnd();
  
  for (it = n->successorsBegin(); it != end; ++it) {
    if (Color::isUncolored((*it)->getColor()) && !(*it)->trivialComponent()) {
      todo_.push(*it);
    }
  }
}

void ImprovedUnfoundedSet::addToSource(Node* n) {
  source_[n->getId()] = 1;
}

}
