#ifdef _MSC_VER
#pragma warning (disable : 4996)
#endif
#include <nomore/nomore_expander.h>
#include <nomore/nomore_program.h>
#include <nomore/nomore_system.h>


#include <platypus/exceptions.h>
#include <platypus/factories/expander_factory.h>
#include <platypus/factories/registration.h>

using namespace NS_NOMORE;

namespace Platypus { namespace Nomore {

Expander::Expander(const ProgramInterface& program, const DelegatableChoice& dc) {
	createStrategy(this, dc);
	program_ = dynamic_cast<const Nomore::Program*>(&program);
	if (!program_) {
		throw TypeError("Nomore-Expander requires Nomore-Program!");
	}
	nomore_.reset(program_->createNomoreSystem());
	if (!dc.empty()) {
		expandBranch(dc);
	}
}

Expander::~Expander() {
}
    
// returns true if the current assignment is not total.
bool Expander::hasUnknown() const {
	return nomore_->hasUnknwon();
}

// returns true if the current assignment is conflicting.
bool Expander::hasConflict() const {
	return nomore_->hasConflict();
}

// returns the current partial assignment restricted to atoms.
PartialAssignment Expander::partialAssignment() const {
  PartialAssignment pa(*program_);
	const Program::CollectionType& atoms = program_->atoms();
	for (Program::CollectionType::size_type i = 0; i < atoms.size(); ++i) {
		Color::ColorValue c = nomore_->getColor(atoms[i]);
		if (c == Color::plus) {
			pa.setTrue(Atom(atoms[i], *program_));
		}
		else if (c == Color::minus) {
			pa.setFalse(Atom(atoms[i], *program_));
		}
	}
	return pa;	
}

// returns the number of atoms that are currently unassigned.
size_t Expander::unknownAtoms() const {
	return nomore_->getNumberOfUnknwonAtoms();	
}

// assert c as choice point and then expand the assignment.
void Expander::strategyExpand(const Choice& choice) {
	choices_.push_back(choice.id());
	if (nomore_->colorChoicePoint(choice)) {
		nomore_->expand();
	}
	else {
		throw InvalidChoice();
	}
}

// reinitializes the expander with a new branch.
void Expander::strategyReinitialize(const DelegatableChoice& dc) {
	undoUntil(0, false);	// undo all choices
	if (!dc.empty()) {
		expandBranch(dc);	
	}
}


// restores everything that was inferred because of choice, then
// calls expand(~choice).
void Expander::strategyBacktrackTo(const Choice& choice) {
	if (nomore_->getColor(choice.id()) == Color::none) {
		throw NoTruthValue();
	}
	if (choice.id() != choices_.back()) {
		throw std::runtime_error("Non-chronological backtracking...!");
	}
	size_t level;
	for (level = choices_.size(); level > 0 && choices_[--level] != choice.id();)
		;
	if (level > choices_.size() || choices_[level] != choice.id()) {
		throw NoChoiceMade();
	}
	undoUntil(level, true);
	nomore_->expand();
}

// returns the current preferred choice point.
// PRE: the assignment is fully expanded and not conflicting.
Choice Expander::strategyGetChoice() {
	return nomore_->select();
}

// returns all possible choices, i.e. all nodes that are currently uncolored.
// Note: since this Expander creates its own choices this method won't be called.
// PRE: the assignment is not conflicting.
PossibleChoices Expander::possibleChoices() const {
	assert(!nomore_->hasConflict() && "NomoreExpander: can not choose from conflicting assignment!");
	PossibleChoices pc;
	size_t max = nomore_->getNumberOfNodes();
	for (size_t i = 0; i != max; ++i) {
		if (nomore_->getColor((AtomId)i) == Color::none) {
			pc.push_back((ChoiceId)i);	
		}
	}
	return pc;
}

// undos every decision level downto (but not including) level.
// If invert is true the choice of level+1 is inverted.
void Expander::undoUntil(size_t level, bool invert) {
	nomore_->reset();
	while (choices_.size() > level + invert && nomore_->restoreLastChoice()) {
		choices_.pop_back();
	}
	if (invert && nomore_->invertLastChoice()) {
		choices_.pop_back();
	}
	assert(choices_.size() == level);
}

// asserts and expands the branch given by the list of choices
// Note: the delegatable choices must be colored as choice points.
// Otherwise the expander could not backtrack the branch
// preventing a later reinitialization of the expander.	
void Expander::expandBranch(const DelegatableChoice& dcl) {
	for (DelegatableChoice::const_iterator it = dcl.begin(); it != dcl.end() && !nomore_->hasConflict(); ++it) {
		// the choice must be either uncolored or a previously learnt fact,
		// i.e. a choice that was inverted on the top-level.
		if (nomore_->colorChoicePoint(*it)) {
			choices_.push_back(it->id());
		}
		else if (!nomore_->hasChoiceColor(*it)) {
			throw InvalidChoice();
		}
	}
	if (!nomore_->hasConflict() && !dcl.empty()) {
		nomore_->expand();	
	}	
}


} }   // end namespace Platypus::Nomore

namespace Platypus {

///////////////////////////////////////////////////////////////////////////////
// factory registration code
///////////////////////////////////////////////////////////////////////////////
static Expander* create(const ProgramInterface& program, const DelegatableChoice& dc) {
	return new Nomore::Expander(program, dc);
}

template<>
void factoryRegistration<Platypus::Nomore::Expander>() {
	ExpanderFactory::instance().add("nomore", &create);
}

}
