/*	$Id: core_callback.h 8138 2007-10-23 12:24:31Z rtichy $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef CORE_CALLBACK_H
#define CORE_CALLBACK_H

#include <platypus/types/choice.h>

namespace Platypus
{
	class PartialAssignment;
	// Callback interface for the CORE implemented by distribution traits.
	// All these methods are only called between a traits classes 
	// setup and teardown methods. Never before or after.
	class CoreDistributionCallback
	{
	public:
		virtual ~CoreDistributionCallback()
		{}
		// Have this method return true when the core should shut down
		// NOTE: This should be a _LOW_ latency method because it 
		// is called alot.
		virtual bool shutdown() = 0;
		// Have this method return true whenever the distribution
		// traits need a delegatable choice for some remote instance
		virtual bool needDelegatableChoice() = 0;
		// The core calls this once the method above returns
		// true and it has a branch ready.
		virtual bool delegate(const DelegatableChoice& dc) = 0;
		// vice versa. Here the core requests a delegatable choice
		// from some remote instance. This call works together with
		// the next so they don't need to be synchronous
		virtual void fileDelegatableChoiceRequest() = 0;
		// Cancel a request for a delegatable choice. The distribution
		// may, in case it already obtained a dc make it available to
		// remote instances or just keep it for subsequent calls to 
		// fileDelegatableChoiceRequest()
		virtual void cancelDelegatableChoiceRequest() = 0;
		// have this method return true whenever a delegatable choice
		// is available to be returned to the core. The method is never
		// called before fileDelegatableChoiceRequest()
		virtual bool hasDelegatableChoice() = 0;
		// return a delegatable choice to the core. The return value
		// is only valid if hasDelegatableChoice() returned true. 
		// otherwise bad things may happen.
		virtual DelegatableChoice delegatableChoice() = 0;
		// Passes an answer set for further processing.
		// NOTE: This method is never called if answer set printing
		// is disabled.
		virtual void answerSet(const PartialAssignment& pa) = 0;
		// Report statistics
		virtual void incExpanderInitializations(size_t inc = 1) = 0;
		virtual void incConflicts(size_t inc = 1) = 0;
		virtual void incBacktracks(size_t inc = 1) = 0;
		virtual void incAnswerSets(size_t inc = 1) = 0;
		virtual void incThreadDelegations(size_t inc = 1) = 0;
		virtual void startExpansion() = 0;
		virtual void finishExpansion() = 0;
		virtual void startExpanderReinitialization() = 0;
		virtual void finishExpanderReinitialization() = 0;
		// Return this processes id. This method exists for debugging
		// reasons mostly. Each instance in platypus should have a quique
		// id that is returned here. The value is completely opaque and
		// absolutely no meaning is implied. The sole purpose is to 
		// be able to differenciate the output of instances. 
		virtual int id() const = 0;
	};
}

#endif
