/*	$Id: smodels_program.cpp 2140 2005-07-03 11:28:23Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cassert>
#include <sstream>
#include <iostream>
#include <algorithm>
#include <platypus/exceptions.h>
#include <platypus/smodels_names.h>
#include <platypus/factories/registration.h>
#include <platypus/factories/program_factory.h>
#include <platypus/smodels/smodels_wrapper.h>
#include <platypus/smodels/smodels_program.h>
#include <smodels/smodels.h>
#include <smodels/program.h>
#include <smodels/atomrule.h>

using namespace std;

#ifdef max
#	undef max
#endif

namespace Platypus
{
	SmodelsEnhancedProgram::~SmodelsEnhancedProgram()
	{
		// quiets compiler warnings about SmodelsHolderFactory
		// having an incomplete type
	}
	SmodelsEnhancedProgram::SmodelsEnhancedProgram()
		:	maxAtomId_(0)
	{
		setup(SMODELS_EMPTY_PROGRAM);
	}
	
	void SmodelsEnhancedProgram::setup(std::istream& input)
	{
		try
		{
			factory_.reset(new SmodelsHolderFactory(input));
		}
		catch(const SmodelsParseError&)
		{
			throw ParseError();
		}
		
		::Program* program = const_cast< ::Program* >(&factory_->smodels()->program);

		atoms_.clear();
		ids_.clear();
		atoms_.reserve(program->number_of_atoms);
		ids_.reserve(factory_->smodels()->program.number_of_atoms);
		maxAtomId_ = 0;

		for(::Node* n = program->atoms.head(); n; n = n->next)
		{
			::Atom* atom = n->atom;

			atoms_.push_back(atom->lparseId_);
			maxAtomId_ = max(maxAtomId_, static_cast<size_t>(atom->lparseId_));
			
			setName(atom->lparseId_, atom->name);
		}
		ids_.resize(maxAtomId_+1);
	}
	void SmodelsEnhancedProgram::setName(size_t n, const char* name)
	{
		if(n >= ids_.size())
			ids_.resize(n+1);
		ids_[n] = name;
	}
	SmodelsHolder* SmodelsEnhancedProgram::holder() const
	{
		return factory_->create();
	}
	ProgramInterface* SmodelsEnhancedProgram::create()
	{
		return new SmodelsEnhancedProgram;
	}
	std::string SmodelsEnhancedProgram::idToName(AtomId id) const
	{
		checkId(id);
		return ids_[id];
	}
	
	const SmodelsEnhancedProgram::CollectionType& SmodelsEnhancedProgram::atoms() const
	{
		return atoms_;
	}
	
	size_t SmodelsEnhancedProgram::numberOfAtoms() const
	{
		return static_cast<size_t>(factory_->smodels()->program.number_of_atoms);
	}
	size_t SmodelsEnhancedProgram::numberOfRules() const
	{
		return static_cast<size_t>(factory_->smodels()->program.number_of_rules);
	}
	
	bool SmodelsEnhancedProgram::hasDisplayName(AtomId id) const
	{
		checkId(id);
		return ids_[id] != 0;
	}
	void SmodelsEnhancedProgram::checkId(AtomId id) const
	{
		if(id == 0 || id > maxAtomId_)
		{
			ostringstream s;
			s << "The atom with id: " << id << " does not exist in the logic program!";
			throw UnknownAtom(s.str());
		}
	}
	template<>
	void factoryRegistration<SmodelsEnhancedProgram>()
	{
		ProgramFactory::instance().add("smodels", SmodelsEnhancedProgram::create);
	}
}

