/*	$Id: expander.h 5947 2006-12-31 15:38:57Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef PLATYPUS_EXPANDER_IMPL_H
#define PLATYPUS_EXPANDER_IMPL_H

#include <memory>
#include <platypus/types/choice.h>
#include <platypus/types/partial_assignment.h>
#include <platypus/types/expander_strategies_callback.h>

namespace Platypus
{
	class BacktrackStrategy;
	class ChooseStrategy;
	namespace ChoicePolicy
	{
		class PolicyBase;
	}

	// Tag interfaces to use with backtracking|choose strategy 
	class Backtrackable {};
	class Choosable {};

	// The smodels encapsulation. An implementation of this
	// interface is required to carry state. This state is
	// refered to as internal partial model.
	class Expander	:	public BacktrackCallback
					,	public ChooseCallback
	{
	public:
		Expander();
		virtual ~Expander();
		// expands the partial model located in the implementation
		// of the Expander with the choice passed.	
		void expand(const Choice& choice);
		
		void backtrackAll();
		
		// backtrack to choice
		void backtrackTo(const Choice& choice);

		// return the next choice either by letting the expander
		// choose or using a specified policy
		Choice getChoice();

		// set the chooser policy to use if the expander
		// does not support choosing itself
		// make copy of policy via clone()
		void setChoicePolicy(const ChoicePolicy::PolicyBase&);
		// assume ownership of pointer
		void setChoicePolicy(ChoicePolicy::PolicyBase*);

		void setManualChoice(bool yes = true);
		
		// reinitializes the expander 
		void reinitialize(const Branch& = Branch());

		bool nativeChooser() const;
		bool nativeBacktracker() const;

		// returns true if the expander has unknown atoms
		virtual bool hasUnknown() const = 0;

		// returns true if the expander has detected a conflict
		virtual bool hasConflict() const = 0;

		// returns a PartialModel representation of the internal
		// partial model. 
		virtual PartialAssignment partialAssignment() const = 0;

		virtual size_t unknownAtoms() const = 0;
	protected:
		// NOTE: This method MUST be called in the c'tor of derived classes so that strategy objects
		// are created!
		void createStrategy(Expander* expander, const DelegatableChoice& dc);
	private:
		virtual void strategyExpand(const Choice& choice) = 0;
		virtual void strategyReinitialize(const DelegatableChoice&) = 0;
		virtual void strategyBacktrackTo(const Choice&) = 0;
		virtual Choice strategyGetChoice() = 0;
		void createChoiceStrategy(Expander* expander);
		void createBracktrackStrategy(Expander* expander, const Branch& dc);
	private:
		std::auto_ptr<ChoicePolicy::PolicyBase> choicePolicy_;
		std::auto_ptr<BacktrackStrategy> backtrackStrategy_;
		std::auto_ptr<ChooseStrategy> chooseStrategy_;
		
	};
}

#endif
