#include <cassert>
#include <cstdlib>
#include <algorithm>
#include <portablethreads/exception.h>
#include <portablethreads/lockfree/heap.h>
#include <portablethreads/lockfree/allocator.h>

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace
		{
			struct OutOfMemory : public std::bad_alloc
			{
				const char *what() const throw()
				{
					return "[PTHeap] Memory allocation failed";
				}
			};
		}
	}
}

#ifdef PT_FAST_MALLOC
#	include <portablethreads/system/fast_malloc_lock_free_heap.cpp>
#else
#	include <portablethreads/system/standard_lock_free_heap.cpp>
#endif

namespace PortableThreads 
{

	namespace LockFree
	{
		namespace Private
		{
			/**************************************************************/
			/* PTAllocatorBase                                            */
			/**************************************************************/
			PTHeap PTAllocatorBase::heap_;
			
		}

		/**************************************************************/
		/* PTHeap                                                     */
		/* The implementation class Private::PTHeapImpl               */
		/* is included above via the define                           */
		/**************************************************************/
		size_t PTHeap::smallestAllocatableBlock()
		{
			return Private::PTHeapImpl::smallestAllocatableBlock();
		}
		size_t PTHeap::largestAllocatableBlock()
		{
			return Private::PTHeapImpl::largestAllocatableBlock();
		}
		


		PTHeap::~PTHeap()
		{}
		PTHeap::PTHeap(AllocateMemory a, FreeMemory r)
			:	pimpl_(0)
		{
			if(!a && !r)
				pimpl_.reset(new Private::PTHeapImpl(&globalOperatorNew, &globalOperatorDelete));
			else if(a && r)
				pimpl_.reset(new Private::PTHeapImpl(a, r));
			else
				throw PTParameterError("[PTHeap] One parameter is a pointer to NULL");
		}
		PTHeap::PTHeap(const PTHeap& other)
			:	pimpl_(new Private::PTHeapImpl(*other.pimpl_))
		{}
		PTHeap& PTHeap::operator=(const PTHeap& other)
		{
			PTHeap(other).swap(*this);
			return *this;
		}
		
		void PTHeap::swap(PTHeap& other)
		{
			pimpl_->swap(*other.pimpl_);
		}
		void* PTHeap::allocate(size_t size)
		{
			return pimpl_->allocate(size);
		}
		void PTHeap::deallocate(void* p)
		{
			pimpl_->deallocate(p);
		}
		void* PTHeap::globalOperatorNew(size_t s)
		{
			return ::operator new(s);
		}
		void PTHeap::globalOperatorDelete(void* p)
		{
			::operator delete(p);
		}

		/**************************************************************/
		/* PTUseHeapMixin                                             */
		/**************************************************************/
		PTHeap* PTUsePTHeapMixin::heap_ = 0;
		void PTUsePTHeapMixin::setHeap(PTHeap& heap)
		{
			heap_ = &heap;
		}
		void* PTUsePTHeapMixin::operator new(size_t size)
		{
			assert(heap_);
			return heap_->allocate(size);
		}
		void PTUsePTHeapMixin::operator delete(void* p, size_t)
		{
			assert(heap_);
			heap_->deallocate(p);
		}
		void* PTUsePTHeapMixin::operator new[](size_t size)
		{
			assert(heap_);
			return heap_->allocate(size);
		}
		void PTUsePTHeapMixin::operator delete[](void* p, size_t)
		{
			assert(heap_);
			heap_->deallocate(p);
		}
		
	}

}

