#include <vector>
#include <cppunit/TestCase.h>
#include <cppunit/TestCaller.h>
#include <cppunit/TestResult.h>
#include <cppunit/TestSuite.h>
#include <cppunit/extensions/HelperMacros.h>
#include <portablethreads/thread.h>
#include <portablethreads/time.h>
#include <portablethreads/semaphore.h>
#include <portablethreads/utility.h>



using namespace std;
using namespace PortableThreads;
using namespace PortableThreads::LockFree;

namespace
{
	class ThreadSilentBase : public PThread
	{
		void unexpectedException() throw()
		{}
	};
	class ThreadBase : public ThreadSilentBase
	{
		void unexpectedException() throw()
		{
			CPPUNIT_ASSERT(false && "no exception excepted");
		}
	};
	
	class Thread0 : public ThreadBase
	{
	public:
		Thread0(PTSemaphore& s, volatile bool& flag)
			:	semaphore_(&s)
			,	flag_(&flag)
		{}
	private:
		void threadMain()
		{
			CPPUNIT_ASSERT_EQUAL(true, semaphore_->tryDown());
			semaphore_->down();
			*flag_ = true;
		}
	private:
		PTSemaphore* semaphore_;
		volatile bool* flag_;
	};
	
	
	class Thread1 : public ThreadBase
	{
	public:
		Thread1(PTSemaphore& sem)
			:	sem_(&sem)
			,	invokations_(0)
			,	shutdown_(false)
		{}
		unsigned invokations() const { return invokations_; }
		void shutdown() { shutdown_ = true; }
	private:
		void threadMain()
		{
			while(true)
			{
				sem_->down();
				if(shutdown_)
					break;
				++invokations_;
			}
		}
	private:
		PTSemaphore* sem_;
		unsigned invokations_;
		volatile bool shutdown_;
	};
}

class SemaphoreTest : public CppUnit::TestFixture
{
public:
	void testDry()
	{
		PTSemaphore s1(0), s2(2);

		CPPUNIT_ASSERT_EQUAL(false, s1.tryDown());
		CPPUNIT_ASSERT_EQUAL(true, s2.tryDown());
		CPPUNIT_ASSERT_EQUAL(true, s2.tryDown());
		
		CPPUNIT_ASSERT_EQUAL(false, s1.tryDown());
		CPPUNIT_ASSERT_EQUAL(false, s2.tryDown());

		s1.up();
		s2.up();
		s1.up();
		s2.up();
		CPPUNIT_ASSERT_EQUAL(true, s1.tryDown());
		CPPUNIT_ASSERT_EQUAL(true, s2.tryDown());
		CPPUNIT_ASSERT_EQUAL(true, s1.tryDown());
		CPPUNIT_ASSERT_EQUAL(true, s2.tryDown());
		CPPUNIT_ASSERT_EQUAL(false, s1.tryDown());
		CPPUNIT_ASSERT_EQUAL(false, s2.tryDown());
	}
	void testSelfBlock()
	{
		PTSemaphore s(1);
		volatile bool f = false;
		volatile bool check;
		Thread0 t(s, f);
		t.run();
		// wait a little, hopefully the thread has started then
		pt_milli_sleep(100);

		// thread should be blocked, hence the flag remains unchanged
		check = false;
		CPPUNIT_ASSERT_EQUAL(check, f);
		

		s.up();
		t.join();

		check = true;
		CPPUNIT_ASSERT_EQUAL(check, f);
	}
	void testThreadIvokations()
	{
		PTSemaphore s;
		const unsigned t = 8;
		const unsigned runs = 1000;
		vector<Thread1*> threads;
		for(unsigned i = 0; i < t; ++i)
		{
			threads.push_back(new Thread1(s));
			threads.back()->run();
		}

		for(unsigned i = 0; i < runs; ++i)
			s.up();

		pt_milli_sleep(100);

		for(unsigned i = 0; i < t; ++i)
			threads[i]->shutdown();

		for(unsigned i = 0; i < t; ++i)
			s.up();

		unsigned sum = 0;
		for(unsigned i = 0; i < t; ++i)
		{
			threads[i]->join();
			sum += threads[i]->invokations();
			delete threads[i];
		}

		// account for remaining sem ups
		while(s.tryDown())
			++sum;

		CPPUNIT_ASSERT_EQUAL(runs, sum);
	}
	CPPUNIT_TEST_SUITE( SemaphoreTest );
		CPPUNIT_TEST( testDry );
		CPPUNIT_TEST( testSelfBlock );
		CPPUNIT_TEST( testThreadIvokations );
	CPPUNIT_TEST_SUITE_END();
};


CPPUNIT_TEST_SUITE_REGISTRATION( SemaphoreTest );



